<?php
/**
 * PayTR Ödeme Gateway Entegrasyonu
 * Mimoza AI SEO Premium için özel geliştirilmiştir
 */

if (!defined('ABSPATH')) exit;

class Mimoza_PayTR_Gateway extends WC_Payment_Gateway {
    
    public function __construct() {
        $this->id = 'mimoza_paytr';
        $this->method_title = 'PayTR (Mimoza)';
        $this->method_description = 'PayTR ile güvenli ödeme - Kredi Kartı, Banka Kartı';
        $this->has_fields = false;
        
        // Ayarları yükle
        $this->init_form_fields();
        $this->init_settings();
        
        // Ayarlardan değerleri al
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->merchant_id = $this->get_option('merchant_id');
        $this->merchant_key = $this->get_option('merchant_key');
        $this->merchant_salt = $this->get_option('merchant_salt');
        $this->test_mode = 'yes' === $this->get_option('test_mode');
        
        // Hooks
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
        add_action('woocommerce_api_mimoza_paytr_callback', [$this, 'handle_callback']);
        add_action('woocommerce_receipt_' . $this->id, [$this, 'receipt_page']);
    }
    
    /**
     * Admin panel ayarları
     */
    public function init_form_fields() {
        $this->form_fields = [
            'enabled' => [
                'title' => 'Aktif/Pasif',
                'type' => 'checkbox',
                'label' => 'PayTR ödeme yöntemini aktifleştir',
                'default' => 'no'
            ],
            'title' => [
                'title' => 'Başlık',
                'type' => 'text',
                'description' => 'Ödeme sayfasında görünecek başlık',
                'default' => 'Kredi Kartı / Banka Kartı (PayTR)',
                'desc_tip' => true
            ],
            'description' => [
                'title' => 'Açıklama',
                'type' => 'textarea',
                'description' => 'Ödeme yöntemi açıklaması',
                'default' => 'Güvenli ödeme için PayTR kullanılmaktadır. Tüm kredi kartları kabul edilir.',
                'desc_tip' => true
            ],
            'merchant_id' => [
                'title' => 'Merchant ID',
                'type' => 'text',
                'description' => 'PayTR üye işyeri numaranız',
                'desc_tip' => true
            ],
            'merchant_key' => [
                'title' => 'Merchant Key',
                'type' => 'password',
                'description' => 'PayTR merchant key',
                'desc_tip' => true
            ],
            'merchant_salt' => [
                'title' => 'Merchant Salt',
                'type' => 'password',
                'description' => 'PayTR merchant salt',
                'desc_tip' => true
            ],
            'test_mode' => [
                'title' => 'Test Modu',
                'type' => 'checkbox',
                'label' => 'Test modunu aktifleştir',
                'default' => 'yes',
                'description' => 'Test modunda gerçek ödeme alınmaz'
            ]
        ];
    }
    
    /**
     * Ödeme işlemi
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        
        // Ödeme sayfasına yönlendir
        return [
            'result' => 'success',
            'redirect' => $order->get_checkout_payment_url(true)
        ];
    }
    
    /**
     * Ödeme sayfası
     */
    public function receipt_page($order_id) {
        $order = wc_get_order($order_id);
        
        // PayTR iframe parametreleri
        $params = $this->prepare_payment_params($order);
        
        if (is_wp_error($params)) {
            wc_add_notice('Ödeme hatası: ' . $params->get_error_message(), 'error');
            return;
        }
        
        ?>
        <div class="mimoza-paytr-payment">
            <h2>🔒 Güvenli Ödeme</h2>
            <p>Ödeme sayfası yükleniyor...</p>
            
            <div id="paytr-iframe-container" style="margin: 30px 0;">
                <iframe 
                    src="<?php echo esc_url($params['iframe_url']); ?>" 
                    id="paytriframe" 
                    frameborder="0" 
                    scrolling="no" 
                    style="width: 100%; height: 800px;">
                </iframe>
            </div>
            
            <script>
            // PayTR iframe otomatik boyutlandırma
            window.addEventListener('message', function(e) {
                if (e.data && e.data.type === 'paytr-iframe-height') {
                    document.getElementById('paytriframe').style.height = e.data.height + 'px';
                }
            });
            </script>
            
            <p class="paytr-info">
                ℹ️ Ödeme işlemi PayTR güvencesi altındadır. Kredi kartı bilgileriniz saklanmaz.
            </p>
        </div>
        
        <style>
        .mimoza-paytr-payment {
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        .mimoza-paytr-payment h2 {
            text-align: center;
            color: #2c3e50;
            margin-bottom: 10px;
        }
        .mimoza-paytr-payment > p {
            text-align: center;
            color: #7f8c8d;
        }
        .paytr-info {
            background: #e8f5e9;
            padding: 15px;
            border-radius: 8px;
            border-left: 4px solid #4caf50;
            margin-top: 20px;
        }
        </style>
        <?php
    }
    
    /**
     * PayTR ödeme parametrelerini hazırla
     */
    private function prepare_payment_params($order) {
        $merchant_id = $this->merchant_id;
        $merchant_key = $this->merchant_key;
        $merchant_salt = $this->merchant_salt;
        
        if (empty($merchant_id) || empty($merchant_key) || empty($merchant_salt)) {
            return new WP_Error('config_error', 'PayTR ayarları eksik!');
        }
        
        // Sepet içeriği
        $basket = [];
        foreach ($order->get_items() as $item) {
            $basket[] = [
                $item->get_name(),
                number_format($item->get_total() / $item->get_quantity(), 2, '.', ''),
                $item->get_quantity()
            ];
        }
        $user_basket = base64_encode(json_encode($basket));
        
        // Müşteri bilgileri
        $email = $order->get_billing_email();
        $payment_amount = $order->get_total() * 100; // Kuruş cinsinden
        $merchant_oid = 'MIMOZA-' . $order->get_id() . '-' . time();
        $user_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
        $user_address = $order->get_billing_address_1();
        $user_phone = $order->get_billing_phone();
        $merchant_ok_url = WC()->api_request_url('mimoza_paytr_callback');
        $merchant_fail_url = WC()->api_request_url('mimoza_paytr_callback');
        $user_ip = $order->get_customer_ip_address();
        
        // Test modu
        $test_mode = $this->test_mode ? 1 : 0;
        
        // Hash token
        $hash_str = $merchant_id . $user_ip . $merchant_oid . $email . $payment_amount . $user_basket . 'no_installment' . '0' . $test_mode . $merchant_salt;
        $paytr_token = base64_encode(hash_hmac('sha256', $hash_str, $merchant_key, true));
        
        // API'ye gönderilecek parametreler
        $post_vals = [
            'merchant_id' => $merchant_id,
            'user_ip' => $user_ip,
            'merchant_oid' => $merchant_oid,
            'email' => $email,
            'payment_amount' => $payment_amount,
            'paytr_token' => $paytr_token,
            'user_basket' => $user_basket,
            'debug_on' => $test_mode,
            'no_installment' => 0,
            'max_installment' => 0,
            'user_name' => $user_name,
            'user_address' => $user_address,
            'user_phone' => $user_phone,
            'merchant_ok_url' => $merchant_ok_url,
            'merchant_fail_url' => $merchant_fail_url,
            'timeout_limit' => 30,
            'currency' => 'TL',
            'test_mode' => $test_mode,
            'lang' => 'tr'
        ];
        
        // Siparişe merchant_oid kaydet
        update_post_meta($order->get_id(), '_paytr_merchant_oid', $merchant_oid);
        
        // PayTR API'sine istek gönder
        $response = wp_remote_post('https://www.paytr.com/odeme/api/get-token', [
            'body' => $post_vals,
            'timeout' => 30
        ]);
        
        if (is_wp_error($response)) {
            return new WP_Error('api_error', 'PayTR bağlantı hatası: ' . $response->get_error_message());
        }
        
        $result = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($result['status'] === 'success') {
            return [
                'iframe_url' => 'https://www.paytr.com/odeme/guvenli/' . $result['token']
            ];
        } else {
            return new WP_Error('token_error', $result['reason'] ?? 'Token alınamadı');
        }
    }
    
    /**
     * PayTR callback (Ödeme sonucu)
     */
    public function handle_callback() {
        // POST verilerini al
        $post_data = $_POST;
        
        if (empty($post_data['merchant_oid'])) {
            wp_die('Invalid callback');
        }
        
        // Siparişi bul
        $merchant_oid = sanitize_text_field($post_data['merchant_oid']);
        
        global $wpdb;
        $order_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_paytr_merchant_oid' AND meta_value = %s",
            $merchant_oid
        ));
        
        if (!$order_id) {
            wp_die('Order not found');
        }
        
        $order = wc_get_order($order_id);
        
        // Hash kontrolü
        $hash = base64_encode(hash_hmac('sha256', $merchant_oid . $this->merchant_salt . $post_data['status'] . $post_data['total_amount'], $this->merchant_key, true));
        
        if ($hash !== $post_data['hash']) {
            wp_die('Invalid hash');
        }
        
        // Ödeme başarılı
        if ($post_data['status'] === 'success') {
            $order->payment_complete();
            $order->add_order_note('PayTR ödeme başarılı. İşlem ID: ' . $post_data['merchant_oid']);
            
            // Başarı sayfasına yönlendir
            wp_redirect($this->get_return_url($order));
            exit;
        } else {
            // Ödeme başarısız
            $order->update_status('failed', 'PayTR ödeme başarısız: ' . ($post_data['failed_reason_msg'] ?? 'Bilinmeyen hata'));
            
            // Sepet sayfasına yönlendir
            wc_add_notice('Ödeme başarısız: ' . ($post_data['failed_reason_msg'] ?? 'Lütfen tekrar deneyin'), 'error');
            wp_redirect(wc_get_checkout_url());
            exit;
        }
    }
}

// Gateway'i WooCommerce'e ekle
add_filter('woocommerce_payment_gateways', 'mimoza_add_paytr_gateway');
function mimoza_add_paytr_gateway($gateways) {
    $gateways[] = 'Mimoza_PayTR_Gateway';
    return $gateways;
}