<?php
/**
 * AJAX İşleyicileri - Ücretsiz Sürüm
 * Sadece temel işlemler
 */

if (!defined('ABSPATH')) exit;

// Link Ekleme
add_action('wp_ajax_mimoza_add_link', 'mimoza_ajax_add_link');
function mimoza_ajax_add_link() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    $anchor = sanitize_text_field($_POST['anchor']);
    $url = esc_url_raw($_POST['url']);
    $type = sanitize_text_field($_POST['type']);
    
    $link_manager = new Mimoza_Link_Manager();
    $result = $link_manager->add_link($anchor, $url, $type);
    
    if ($result) {
        wp_send_json_success('Link eklendi');
    } else {
        wp_send_json_error('Link eklenemedi');
    }
}

// Link Silme
add_action('wp_ajax_mimoza_delete_link', 'mimoza_ajax_delete_link');
function mimoza_ajax_delete_link() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    $index = intval($_POST['index']);
    $type = sanitize_text_field($_POST['type']);
    
    $link_manager = new Mimoza_Link_Manager();
    $result = $link_manager->delete_link($index, $type);
    
    if ($result) {
        wp_send_json_success('Link silindi');
    } else {
        wp_send_json_error('Link silinemedi');
    }
}

// Link Export
add_action('wp_ajax_mimoza_export_links', 'mimoza_ajax_export_links');
function mimoza_ajax_export_links() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_die('Yetkiniz yok');
    }
    
    $link_manager = new Mimoza_Link_Manager();
    $type = sanitize_text_field($_GET['type'] ?? 'all');
    $link_manager->export_links_to_csv($type);
}

// CSV İçe Aktarma
add_action('wp_ajax_mimoza_import_csv', 'mimoza_ajax_import_csv');
function mimoza_ajax_import_csv() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    if (empty($_FILES['csv_file'])) {
        wp_send_json_error('Dosya yüklenmedi');
    }
    
    $file = $_FILES['csv_file'];
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        wp_send_json_error('Dosya yükleme hatası');
    }
    
    if (pathinfo($file['name'], PATHINFO_EXTENSION) !== 'csv') {
        wp_send_json_error('Sadece CSV dosyası yüklenebilir');
    }
    
    $link_manager = new Mimoza_Link_Manager();
    $imported = $link_manager->import_links_from_csv($file['tmp_name'], 'internal');
    
    if (is_wp_error($imported)) {
        wp_send_json_error($imported->get_error_message());
    }
    
    wp_send_json_success([
        'message' => $imported . ' link başarıyla içe aktarıldı'
    ]);
}

// API Test
add_action('wp_ajax_mimoza_test_api', 'mimoza_ajax_test_api');
function mimoza_ajax_test_api() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    $api_type = sanitize_text_field($_POST['api_type']);
    $api_key = sanitize_text_field($_POST['api_key']);
    
    if (empty($api_key)) {
        wp_send_json_error('API anahtarı boş');
    }
    
    $result = false;
    
    switch ($api_type) {
        case 'openai':
            $response = wp_remote_get('https://api.openai.com/v1/models', [
                'headers' => ['Authorization' => 'Bearer ' . $api_key],
                'timeout' => 10
            ]);
            $result = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
            break;
            
        case 'claude':
            $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
                'headers' => [
                    'x-api-key' => $api_key,
                    'anthropic-version' => '2023-06-01',
                    'Content-Type' => 'application/json'
                ],
                'body' => json_encode([
                    'model' => 'claude-3-haiku-20240307',
                    'max_tokens' => 10,
                    'messages' => [['role' => 'user', 'content' => 'test']]
                ]),
                'timeout' => 10
            ]);
            $result = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
            break;
            
        case 'gemini':
            $response = wp_remote_get(
                'https://generativelanguage.googleapis.com/v1beta/models?key=' . $api_key,
                ['timeout' => 10]
            );
            $result = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
            break;
            
        case 'pexels':
            $response = wp_remote_get('https://api.pexels.com/v1/search?query=test&per_page=1', [
                'headers' => ['Authorization' => $api_key],
                'timeout' => 10
            ]);
            $result = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
            break;
            
        case 'unsplash':
            $response = wp_remote_get('https://api.unsplash.com/photos/random?count=1', [
                'headers' => ['Authorization' => 'Client-ID ' . $api_key],
                'timeout' => 10
            ]);
            $result = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
            break;
            
        case 'pixabay':
            $response = wp_remote_get(
                'https://pixabay.com/api/?key=' . $api_key . '&q=test&per_page=1',
                ['timeout' => 10]
            );
            $result = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
            break;
    }
    
    if ($result) {
        wp_send_json_success('API bağlantısı başarılı!');
    } else {
        wp_send_json_error('API bağlantısı başarısız. Lütfen anahtarınızı kontrol edin.');
    }
}

// Zamanlanmış post'u hemen yayınla
add_action('wp_ajax_mimoza_publish_now', 'mimoza_ajax_publish_now');
function mimoza_ajax_publish_now() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('publish_posts')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    $post_id = intval($_POST['post_id']);
    
    wp_update_post([
        'ID' => $post_id,
        'post_status' => 'publish',
        'post_date' => current_time('mysql'),
        'post_date_gmt' => current_time('mysql', 1)
    ]);
    
    delete_post_meta($post_id, '_mimoza_scheduled');
    
    wp_send_json_success('Makale yayınlandı');
}

// Zamanlamayı iptal et
add_action('wp_ajax_mimoza_cancel_schedule', 'mimoza_ajax_cancel_schedule');
function mimoza_ajax_cancel_schedule() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('edit_posts')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    $post_id = intval($_POST['post_id']);
    
    wp_update_post([
        'ID' => $post_id,
        'post_status' => 'draft'
    ]);
    
    delete_post_meta($post_id, '_mimoza_scheduled');
    
    wp_send_json_success('Zamanlama iptal edildi');
}

// Tümünü hemen yayınla
add_action('wp_ajax_mimoza_publish_all_scheduled', 'mimoza_ajax_publish_all_scheduled');
function mimoza_ajax_publish_all_scheduled() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('publish_posts')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    global $wpdb;
    $posts = $wpdb->get_col("
        SELECT p.ID
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE p.post_status = 'future'
        AND pm.meta_key = '_mimoza_scheduled'
    ");
    
    foreach ($posts as $post_id) {
        wp_update_post([
            'ID' => $post_id,
            'post_status' => 'publish',
            'post_date' => current_time('mysql'),
            'post_date_gmt' => current_time('mysql', 1)
        ]);
        
        delete_post_meta($post_id, '_mimoza_scheduled');
    }
    
    wp_send_json_success([
        'message' => count($posts) . ' makale yayınlandı'
    ]);
}

// Tüm zamanlamaları iptal et
add_action('wp_ajax_mimoza_cancel_all_schedules', 'mimoza_ajax_cancel_all_schedules');
function mimoza_ajax_cancel_all_schedules() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('edit_posts')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    global $wpdb;
    $posts = $wpdb->get_col("
        SELECT p.ID
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE p.post_status = 'future'
        AND pm.meta_key = '_mimoza_scheduled'
    ");
    
    foreach ($posts as $post_id) {
        wp_update_post([
            'ID' => $post_id,
            'post_status' => 'draft'
        ]);
        
        delete_post_meta($post_id, '_mimoza_scheduled');
    }
    
    wp_send_json_success([
        'message' => count($posts) . ' zamanlama iptal edildi'
    ]);
}

// İstatistikler
add_action('wp_ajax_mimoza_get_stats', 'mimoza_ajax_get_stats');
function mimoza_ajax_get_stats() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Yetkiniz yok');
    }
    
    global $wpdb;
    $table = $wpdb->prefix . 'mimoza_generations';
    
    $period = sanitize_text_field($_POST['period'] ?? 'month');
    
    $where = "1=1";
    switch ($period) {
        case 'today':
            $where = "DATE(created_at) = CURDATE()";
            break;
        case 'week':
            $where = "created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
            break;
        case 'month':
            $where = "created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
            break;
        case 'year':
            $where = "YEAR(created_at) = YEAR(NOW())";
            break;
    }
    
    $stats = [
        'total' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE $where"),
        'by_model' => $wpdb->get_results("
            SELECT model, COUNT(*) as count, AVG(word_count) as avg_words
            FROM $table 
            WHERE $where
            GROUP BY model
            ORDER BY count DESC
        "),
        'by_date' => $wpdb->get_results("
            SELECT DATE(created_at) as date, COUNT(*) as count
            FROM $table
            WHERE $where
            GROUP BY DATE(created_at)
            ORDER BY date DESC
            LIMIT 30
        ")
    ];
    
    wp_send_json_success($stats);
}

// Duplicate Checker
add_action('wp_ajax_mimoza_check_duplicates', 'mimoza_ajax_check_duplicates');
function mimoza_ajax_check_duplicates() {
    check_ajax_referer('mimoza_nonce', 'nonce');
    
    $keywords = array_filter(array_map('trim', explode("\n", $_POST['keywords'])));
    $duplicates = [];
    
    foreach ($keywords as $keyword) {
        $existing = get_posts([
            'post_type' => 'post',
            'title' => $keyword,
            'posts_per_page' => 1
        ]);
        
        if (!empty($existing)) {
            $duplicates[] = [
                'keyword' => $keyword,
                'post_id' => $existing[0]->ID,
                'edit_link' => get_edit_post_link($existing[0]->ID),
                'view_link' => get_permalink($existing[0]->ID)
            ];
        }
    }
    
    wp_send_json_success([
        'has_duplicates' => !empty($duplicates),
        'duplicates' => $duplicates
    ]);
}