<?php
/**
 * Veritabanı Kurulum - Ücretsiz Sürüm
 * Lisans tablosu yok
 */

if (!defined('ABSPATH')) exit;

class Mimoza_Database_Setup {
    
    public static function install() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Generations tablosu
        $sql_generations = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}mimoza_generations (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            keyword varchar(255) NOT NULL,
            model varchar(100) NOT NULL,
            word_count int NOT NULL DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY post_id (post_id),
            KEY keyword (keyword),
            KEY created_at (created_at),
            KEY model (model)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_generations);
        
        // Varsayılan ayarları ekle
        self::set_default_options();
        
        // Cron job'ları kur
        self::schedule_cron_jobs();
        
        // Veritabanı versiyonunu kaydet
        update_option('mimoza_db_version', MIMOZA_VERSION);
    }
    
    public static function set_default_options() {
        add_option('mimoza_version', MIMOZA_VERSION);
        add_option('mimoza_installed_at', current_time('mysql'));
        add_option('mimoza_internal_links', []);
        add_option('mimoza_external_links', []);
    }
    
    public static function schedule_cron_jobs() {
        // Zamanlanmış post kontrolü (her saat)
        if (!wp_next_scheduled('mimoza_check_scheduled_posts')) {
            wp_schedule_event(time(), 'hourly', 'mimoza_check_scheduled_posts');
        }
        
        // Log temizleme (haftalık)
        if (!wp_next_scheduled('mimoza_cleanup_logs')) {
            wp_schedule_event(time(), 'weekly', 'mimoza_cleanup_logs');
        }
    }
    
    public static function uninstall() {
        // Kullanıcı isterse veritabanını sil
        if (get_option('mimoza_delete_data_on_uninstall')) {
            global $wpdb;
            
            $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}mimoza_generations");
            
            // Options temizle
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'mimoza_%'");
        }
        
        // Cron job'ları temizle
        wp_clear_scheduled_hook('mimoza_check_scheduled_posts');
        wp_clear_scheduled_hook('mimoza_cleanup_logs');
    }
    
    public static function upgrade() {
        $current_version = get_option('mimoza_db_version', '1.0');
        
        if (version_compare($current_version, MIMOZA_VERSION, '<')) {
            self::install();
            update_option('mimoza_db_version', MIMOZA_VERSION);
        }
    }
    
    /**
     * Eski kayıtları temizle
     */
    public static function cleanup_old_logs() {
        global $wpdb;
        
        // 1 yıldan eski generation kayıtlarını sil
        $wpdb->query("
            DELETE FROM {$wpdb->prefix}mimoza_generations 
            WHERE created_at < DATE_SUB(NOW(), INTERVAL 1 YEAR)
        ");
    }
    
    /**
     * İstatistikler
     */
    public static function get_statistics() {
        global $wpdb;
        
        return [
            'total_generations' => $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}mimoza_generations"),
            'last_generation' => $wpdb->get_var("SELECT MAX(created_at) FROM {$wpdb->prefix}mimoza_generations")
        ];
    }
}

// Cron event handlers
add_action('mimoza_cleanup_logs', ['Mimoza_Database_Setup', 'cleanup_old_logs']);

// Zamanlanmış post kontrolü
add_action('mimoza_check_scheduled_posts', 'mimoza_process_scheduled_posts');
function mimoza_process_scheduled_posts() {
    global $wpdb;
    
    $now = current_time('mysql');
    
    // Future status'teki ve mimoza tarafından zamanlanmış post'ları bul
    $scheduled = $wpdb->get_results($wpdb->prepare("
        SELECT p.ID 
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE p.post_status = 'future'
        AND p.post_date <= %s
        AND pm.meta_key = '_mimoza_scheduled'
        LIMIT 10
    ", $now));
    
    foreach ($scheduled as $item) {
        // Post'u yayınla
        wp_update_post([
            'ID' => $item->ID,
            'post_status' => 'publish',
            'post_date' => $now,
            'post_date_gmt' => get_gmt_from_date($now)
        ]);
        
        delete_post_meta($item->ID, '_mimoza_scheduled');
    }
}