<?php
/**
 * Gelişmiş AI Motor Entegrasyonu
 * ChatGPT-5, Claude Sonnet 4.5, Gemini 2.0, Jasper
 * 
 * @version 6.0
 */

if (!defined('ABSPATH')) exit;

class Mimoza_Advanced_AI_Engine {
    
    private $link_manager;
    private $seo_optimizer;
    
    // Desteklenen AI Modelleri
    private $supported_models = [
        'openai' => [
            'gpt-4o' => 'GPT-4o (En Gelişmiş)',
            'gpt-4-turbo' => 'GPT-4 Turbo (Hızlı & Güçlü)',
            'gpt-4' => 'GPT-4 (Yüksek Kalite)',
            'gpt-3.5-turbo' => 'GPT-3.5 Turbo (Ekonomik)'
        ],
        'anthropic' => [
            'claude-sonnet-4-5-20250929' => 'Claude Sonnet 4.5 (En Akıllı)',
            'claude-3-5-sonnet-20241022' => 'Claude 3.5 Sonnet (Dengeli)',
            'claude-3-opus-20240229' => 'Claude 3 Opus (Maksimum Kalite)',
            'claude-3-sonnet-20240229' => 'Claude 3 Sonnet (Hızlı)',
            'claude-3-haiku-20240307' => 'Claude 3 Haiku (Çok Hızlı)'
        ],
        'google' => [
            'gemini-2.0-flash-exp' => 'Gemini 2.0 Flash (Deneysel)',
            'gemini-1.5-pro' => 'Gemini 1.5 Pro (Gelişmiş)',
            'gemini-1.5-flash' => 'Gemini 1.5 Flash (Hızlı)',
            'gemini-pro' => 'Gemini Pro (Standart)'
        ],
        'jasper' => [
            'jasper-pro' => 'Jasper Pro (Premium)',
            'jasper-standard' => 'Jasper Standard'
        ]
    ];
    
    public function __construct() {
        $this->link_manager = new Mimoza_Link_Manager();
        $this->seo_optimizer = new Mimoza_SEO_Optimizer();
    }
    
    /**
     * Gelişmiş İçerik Üretimi - Ana Method
     */
    public function generate_content($keyword, $settings) {
        // 1. Prompt hazırla (kullanıcı niyeti analizi)
        $prompt = $this->build_intelligent_prompt($keyword, $settings);
        
        // 2. Model seç ve çağır
        $raw_content = $this->call_ai_model($prompt, $settings['model']);
        
        if (is_wp_error($raw_content)) {
            return $raw_content;
        }
        
        // 3. İçeriği zenginleştir
        $content = $this->enrich_content($raw_content, $keyword, $settings);
        
        // 4. Internal & External link ekle
        $content = $this->add_intelligent_links($content, $keyword);
        
        // 5. SEO yapısı ekle (H1-H3, paragraflar)
        $content = $this->structure_content($content);
        
        return $content;
    }
    
    /**
     * Akıllı Prompt Oluşturma Sistemi
     */
    private function build_intelligent_prompt($keyword, $settings) {
        $word_count = $settings['word_count'] ?? 2000;
        $custom_prompt = $settings['custom_prompt'] ?? '';
        $language = $settings['language'] ?? 'tr';
        
        // Kullanıcı özel prompt verdiyse onu kullan
        if (!empty($custom_prompt)) {
            // {keyword} placeholder'ını değiştir
            $final_prompt = str_replace('{keyword}', $keyword, $custom_prompt);
            
            // Ek yönergeler ekle
            $final_prompt .= "\n\nÖNEMLİ TALİMATLAR:\n";
            $final_prompt .= "- Yaklaşık {$word_count} kelime uzunluğunda olsun\n";
            $final_prompt .= "- HTML formatında yaz (h2, h3, p, strong, em etiketleri kullan)\n";
            $final_prompt .= "- SEO uyumlu ve özgün içerik üret\n";
            $final_prompt .= "- Başlıkları mantıklı sırada kullan (h2 > h3 > h4)\n";
            
            return $final_prompt;
        }
        
        // Varsayılan gelişmiş SEO prompt
        return $this->create_advanced_seo_prompt($keyword, $word_count, $language);
    }
    
    /**
     * Gelişmiş SEO Odaklı Prompt
     */
    private function create_advanced_seo_prompt($keyword, $word_count, $language = 'tr') {
        $prompts = [
            'tr' => "Sen bir SEO uzmanı ve profesyonel içerik yazarısın. Aşağıdaki anahtar kelime hakkında detaylı, özgün ve SEO uyumlu bir makale yaz.

ANAHTAR KELİME: {$keyword}

MAKALE GEREKSİNİMLERİ:
📏 Uzunluk: Yaklaşık {$word_count} kelime
🎯 Amaç: Okuyucuya maksimum değer sunmak ve Google'da üst sıralarda yer almak
✍️ Ton: Profesyonel ama samimi, bilgilendirici ve anlaşılır

İÇERİK YAPISI:
1. GİRİŞ (150-200 kelime)
   - Konuyu tanıt
   - Makalenin okuyucuya sağlayacağı faydaları belirt
   - Merak uyandırıcı bir hook kullan

2. ANA İÇERİK (5-7 ana bölüm)
   Her bölümde:
   - <h2> başlığı kullan (SEO dostu, anahtar kelime içerir)
   - Alt başlıklar için <h3> kullan
   - Her paragraf 3-5 cümle olsun
   - Maddeler halinde bilgi ver (gerektiğinde <ul> <li>)
   - Önemli noktaları <strong> ile vurgula
   - Örnekler ve gerçek hayat senaryoları ekle

3. SONUÇ (100-150 kelime)
   - Ana noktaları özetle
   - Okuyucuya eylem önerisi sun (CTA)
   - Konuyla ilgili geleceğe dönük bakış

ANAHTAR KELİME KULLANIMI:
- İlk 100 kelimede en az 1 kez geçmeli
- Doğal ve akıcı şekilde içeriğe yayılmalı
- Başlıklarda stratejik kullanım
- LSI anahtar kelimeleri (ilgili terimler) dahil et

SEO TALİMATLARI:
✓ Özgün içerik (kopyala-yapıştır YOK)
✓ Anahtar kelime yoğunluğu: %1-2
✓ Uzun kuyruk anahtar kelimeleri kullan
✓ E-A-T prensiplerine uy (Expertise, Authority, Trust)
✓ Okuma kolaylığı: Flesch Reading Ease skoru 60+

FORMATLAMA:
- HTML etiketleri kullan: <p>, <h2>, <h3>, <strong>, <em>, <ul>, <li>
- Her paragraf ayrı <p> etiketi içinde
- Başlıklar hiyerarşik sırada (h2 > h3)
- Uzun metinleri madde işaretleriyle böl

ÖNEMLİ NOTLAR:
❌ YAPMA: Gereksiz doldurma, aşırı tekrar, robot gibi yazma
✅ YAP: Değerli bilgi sun, kullanıcı deneyimini önceliklendir, özgün ol

Şimdi yukarıdaki talimatlara göre makalenin tamamını yaz. Direkt içerikle başla (meta bilgiler ekleme).",

            'en' => "You are an SEO expert and professional content writer. Write a detailed, unique and SEO-friendly article about the following keyword.

KEYWORD: {$keyword}

ARTICLE REQUIREMENTS:
📏 Length: Approximately {$word_count} words
🎯 Purpose: Provide maximum value to readers and rank high on Google
✍️ Tone: Professional yet friendly, informative and clear

CONTENT STRUCTURE:
1. INTRODUCTION (150-200 words)
   - Introduce the topic
   - State the benefits the article will provide
   - Use an attention-grabbing hook

2. MAIN CONTENT (5-7 main sections)
   For each section:
   - Use <h2> heading (SEO-friendly, includes keyword)
   - Use <h3> for subheadings
   - Each paragraph 3-5 sentences
   - Use bullet points when needed (<ul> <li>)
   - Emphasize important points with <strong>
   - Include examples and real-life scenarios

3. CONCLUSION (100-150 words)
   - Summarize main points
   - Provide a call to action
   - Forward-looking perspective

KEYWORD USAGE:
- Must appear at least once in first 100 words
- Naturally distributed throughout content
- Strategic use in headings
- Include LSI keywords (related terms)

SEO INSTRUCTIONS:
✓ Original content (NO copy-paste)
✓ Keyword density: 1-2%
✓ Use long-tail keywords
✓ Follow E-A-T principles (Expertise, Authority, Trust)
✓ Readability: Flesch Reading Ease score 60+

FORMATTING:
- Use HTML tags: <p>, <h2>, <h3>, <strong>, <em>, <ul>, <li>
- Each paragraph in separate <p> tag
- Hierarchical headings (h2 > h3)
- Break long text with bullet points

IMPORTANT NOTES:
❌ DON'T: Unnecessary stuffing, excessive repetition, robotic writing
✅ DO: Provide valuable information, prioritize user experience, be original

Now write the complete article according to these instructions. Start directly with content (no meta information)."
        ];
        
        $template = $prompts[$language] ?? $prompts['tr'];
        
        return str_replace(['{keyword}', '{$word_count}'], [$keyword, $word_count], $template);
    }
    
    /**
     * Model çağırma - Merkezi router
     */
    private function call_ai_model($prompt, $model) {
        // Model provider'ı belirle
        $provider = $this->get_model_provider($model);
        
        switch ($provider) {
            case 'openai':
                return $this->call_openai($prompt, $model);
            case 'anthropic':
                return $this->call_anthropic($prompt, $model);
            case 'google':
                return $this->call_google($prompt, $model);
            case 'jasper':
                return $this->call_jasper($prompt, $model);
            default:
                return new WP_Error('invalid_provider', 'Geçersiz AI provider');
        }
    }
    
    /**
     * Model provider'ını belirle
     */
    private function get_model_provider($model) {
        if (strpos($model, 'gpt') !== false) return 'openai';
        if (strpos($model, 'claude') !== false) return 'anthropic';
        if (strpos($model, 'gemini') !== false) return 'google';
        if (strpos($model, 'jasper') !== false) return 'jasper';
        
        return 'unknown';
    }
    
    /**
     * OpenAI API Çağrısı (GPT-4o, GPT-4 Turbo vb.)
     */
    private function call_openai($prompt, $model) {
        $api_key = get_option('mimoza_openai_api_key');
        
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'OpenAI API anahtarı eksik');
        }
        
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ],
            'body' => json_encode([
                'model' => $model,
                'messages' => [
                    ['role' => 'system', 'content' => 'Sen profesyonel bir SEO içerik yazarısın.'],
                    ['role' => 'user', 'content' => $prompt]
                ],
                'temperature' => 0.7,
                'max_tokens' => 4096
            ]),
            'timeout' => 180
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($data['choices'][0]['message']['content'])) {
            return $data['choices'][0]['message']['content'];
        }
        
        return new WP_Error('api_error', 'OpenAI yanıt vermedi');
    }
    
    /**
     * Anthropic Claude API Çağrısı (Claude Sonnet 4.5 vb.)
     */
    private function call_anthropic($prompt, $model) {
        $api_key = get_option('mimoza_claude_api_key');
        
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'Claude API anahtarı eksik');
        }
        
        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01'
            ],
            'body' => json_encode([
                'model' => $model,
                'max_tokens' => 4096,
                'messages' => [
                    ['role' => 'user', 'content' => $prompt]
                ]
            ]),
            'timeout' => 180
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($data['content'][0]['text'])) {
            return $data['content'][0]['text'];
        }
        
        return new WP_Error('api_error', 'Claude yanıt vermedi');
    }
    
    /**
     * Google Gemini API Çağrısı (Gemini 2.0 Flash vb.)
     */
    private function call_google($prompt, $model) {
        $api_key = get_option('mimoza_gemini_api_key');
        
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'Gemini API anahtarı eksik');
        }
        
        $response = wp_remote_post(
            'https://generativelanguage.googleapis.com/v1beta/models/' . $model . ':generateContent?key=' . $api_key,
            [
                'headers' => ['Content-Type' => 'application/json'],
                'body' => json_encode([
                    'contents' => [
                        ['parts' => [['text' => $prompt]]]
                    ],
                    'generationConfig' => [
                        'maxOutputTokens' => 8192,
                        'temperature' => 0.7
                    ]
                ]),
                'timeout' => 180
            ]
        );
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($data['candidates'][0]['content']['parts'][0]['text'])) {
            return $data['candidates'][0]['content']['parts'][0]['text'];
        }
        
        return new WP_Error('api_error', 'Gemini yanıt vermedi');
    }
    
    /**
     * Jasper API Çağrısı
     */
    private function call_jasper($prompt, $model) {
        $api_key = get_option('mimoza_jasper_api_key');
        
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'Jasper API anahtarı eksik');
        }
        
        // Jasper API endpoint'i
        $response = wp_remote_post('https://api.jasper.ai/v1/completions', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ],
            'body' => json_encode([
                'model' => $model,
                'prompt' => $prompt,
                'max_tokens' => 4000
            ]),
            'timeout' => 180
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($data['choices'][0]['text'])) {
            return $data['choices'][0]['text'];
        }
        
        return new WP_Error('api_error', 'Jasper yanıt vermedi');
    }
    
    /**
     * İçeriği Zenginleştir
     */
    private function enrich_content($content, $keyword, $settings) {
        // Schema markup ekle (Article)
        $schema = $this->generate_article_schema($keyword, $content);
        
        // FAQ Schema ekle (varsa)
        if ($this->has_faq_structure($content)) {
            $schema .= $this->generate_faq_schema($content);
        }
        
        // Metni temizle ve düzenle
        $content = $this->clean_and_format($content);
        
        return $content . "\n\n" . $schema;
    }
    
    /**
     * Akıllı Internal & External Linking
     */
    private function add_intelligent_links($content, $keyword) {
        // 1. Internal linkler ekle
        $content = $this->add_internal_links($content, $keyword);
        
        // 2. Yüksek otoriteli external linkler ekle
        $content = $this->add_authority_external_links($content, $keyword);
        
        return $content;
    }
    
    /**
     * Internal Link Ekleme
     */
    private function add_internal_links($content, $keyword) {
        global $wpdb;
        
        // Site içi ilgili makaleleri bul
        $related_posts = $wpdb->get_results($wpdb->prepare("
            SELECT ID, post_title 
            FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND post_type = 'post'
            AND post_title LIKE %s
            ORDER BY post_date DESC
            LIMIT 5
        ", '%' . $wpdb->esc_like($keyword) . '%'));
        
        // Link havuzundan da ekle
        $link_pool = $this->link_manager->get_all_links();
        $internal_links = $link_pool['internal'] ?? [];
        
        // 3-5 internal link ekle
        $added_count = 0;
        $max_links = rand(3, 5);
        
        // Önce ilgili postlardan
        foreach ($related_posts as $post) {
            if ($added_count >= $max_links) break;
            
            $anchor = wp_trim_words($post->post_title, 4);
            $url = get_permalink($post->ID);
            
            // İçerikte bu anchor text varsa linkle
            if (stripos($content, $anchor) !== false) {
                $content = preg_replace(
                    '/\b' . preg_quote($anchor, '/') . '\b/i',
                    '<a href="' . $url . '">' . $anchor . '</a>',
                    $content,
                    1
                );
                $added_count++;
            }
        }
        
        // Link havuzundan da ekle
        foreach ($internal_links as $link) {
            if ($added_count >= $max_links) break;
            
            $anchor = $link['anchor'];
            $url = $link['url'];
            
            if (stripos($content, $anchor) !== false) {
                $content = preg_replace(
                    '/\b' . preg_quote($anchor, '/') . '\b/i',
                    '<a href="' . $url . '">' . $anchor . '</a>',
                    $content,
                    1
                );
                $added_count++;
            }
        }
        
        return $content;
    }
    
    /**
     * Yüksek Otoriteli External Linkler Ekle
     */
    private function add_authority_external_links($content, $keyword) {
        // Otoriteli siteler
        $authority_sites = [
            'https://developers.google.com/search',
            'https://en.wikipedia.org',
            'https://tr.wikipedia.org',
            'https://ahrefs.com/blog',
            'https://moz.com/blog',
            'https://search.google.com/search-console'
        ];
        
        // Link havuzundan external linkleri al
        $link_pool = $this->link_manager->get_all_links();
        $external_links = $link_pool['external'] ?? [];
        
        $added_count = 0;
        $max_external = rand(1, 3);
        
        // Link havuzundan ekle
        foreach ($external_links as $link) {
            if ($added_count >= $max_external) break;
            
            $anchor = $link['anchor'];
            $url = $link['url'];
            
            if (stripos($content, $anchor) !== false) {
                $content = preg_replace(
                    '/\b' . preg_quote($anchor, '/') . '\b/i',
                    '<a href="' . $url . '" rel="nofollow noopener" target="_blank">' . $anchor . '</a>',
                    $content,
                    1
                );
                $added_count++;
            }
        }
        
        // Eğer yeterli link eklenmemişse, genel otoriteli bir link ekle
        if ($added_count < 1) {
            $random_authority = $authority_sites[array_rand($authority_sites)];
            
            // İlk paragrafın sonuna ekle
            $paragraphs = explode('</p>', $content, 3);
            if (count($paragraphs) >= 2) {
                $paragraphs[1] .= ' <a href="' . $random_authority . '" rel="nofollow noopener" target="_blank">Daha fazla bilgi için tıklayın</a>.</p>';
                $content = implode('</p>', $paragraphs);
            }
        }
        
        return $content;
    }
    
    /**
     * İçeriği Yapılandır (H1-H3, Paragraflar)
     */
    private function structure_content($content) {
        // Eğer içerik zaten HTML yapısındaysa, sadece optimize et
        if (preg_match('/<h[1-6]>/i', $content)) {
            return $this->optimize_html_structure($content);
        }
        
        // Markdown'dan HTML'e çevir
        $content = $this->markdown_to_html($content);
        
        return $content;
    }
    
    /**
     * HTML Yapısını Optimize Et
     */
    private function optimize_html_structure($content) {
        // H1 varsa H2'ye çevir (WordPress post title zaten H1)
        $content = preg_replace('/<h1[^>]*>(.*?)<\/h1>/i', '<h2>$1</h2>', $content);
        
        // Boş paragrafları temizle
        $content = preg_replace('/<p>\s*<\/p>/', '', $content);
        
        // Her paragrafın düzgün kapandığından emin ol
        $content = preg_replace('/<p>([^<]*?)(?=<(?:h[2-6]|p|ul|ol|div|$))/i', '<p>$1</p>', $content);
        
        return $content;
    }
    
    /**
     * Markdown to HTML
     */
    private function markdown_to_html($text) {
        // Başlıklar
        $text = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $text);
        $text = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $text);
        $text = preg_replace('/^# (.+)$/m', '<h2>$1</h2>', $text);
        
        // Bold
        $text = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $text);
        
        // Italic
        $text = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $text);
        
        // Listeler
        $text = preg_replace('/^\* (.+)$/m', '<li>$1</li>', $text);
        $text = preg_replace('/(<li>.*<\/li>)/s', '<ul>$1</ul>', $text);
        
        // Paragraflar
        $lines = explode("\n\n", $text);
        $formatted = '';
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;
            
            if (!preg_match('/^<(h[2-6]|ul|ol)/', $line)) {
                $formatted .= '<p>' . $line . '</p>' . "\n";
            } else {
                $formatted .= $line . "\n";
            }
        }
        
        return $formatted;
    }
    
    /**
     * Article Schema Markup Üret
     */
    private function generate_article_schema($title, $content) {
        $schema = '<script type="application/ld+json">' . "\n";
        $schema .= json_encode([
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            'headline' => $title,
            'description' => wp_trim_words(wp_strip_all_tags($content), 30),
            'author' => [
                '@type' => 'Person',
                'name' => get_bloginfo('name')
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => get_bloginfo('name'),
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => get_site_icon_url()
                ]
            ],
            'datePublished' => current_time('c'),
            'dateModified' => current_time('c')
        ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        $schema .= "\n</script>\n";
        
        return $schema;
    }
    
    /**
     * FAQ Schema Üret
     */
    private function generate_faq_schema($content) {
        // FAQ yapısını tespit et
        preg_match_all('/<h3>(.*?)<\/h3>\s*<p>(.*?)<\/p>/si', $content, $matches, PREG_SET_ORDER);
        
        if (empty($matches)) {
            return '';
        }
        
        $faq_items = [];
        foreach ($matches as $match) {
            $faq_items[] = [
                '@type' => 'Question',
                'name' => strip_tags($match[1]),
                'acceptedAnswer' => [
                    '@type' => 'Answer',
                    'text' => strip_tags($match[2])
                ]
            ];
        }
        
        $schema = '<script type="application/ld+json">' . "\n";
        $schema .= json_encode([
            '@context' => 'https://schema.org',
            '@type' => 'FAQPage',
            'mainEntity' => $faq_items
        ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        $schema .= "\n</script>\n";
        
        return $schema;
    }
    
    /**
     * FAQ Yapısı Kontrolü
     */
    private function has_faq_structure($content) {
        return preg_match('/<h3>.*?<\/h3>\s*<p>.*?<\/p>/si', $content);
    }
    
    /**
     * İçeriği Temizle ve Düzenle
     */
    private function clean_and_format($content) {
        // Gereksiz boşlukları temizle
        $content = preg_replace('/\s+/', ' ', $content);
        
        // Çift satır sonlarını düzelt
        $content = preg_replace('/\n\s*\n/', "\n\n", $content);
        
        // Başlangıç ve sondaki boşlukları temizle
        $content = trim($content);
        
        return $content;
    }
    
    /**
     * Desteklenen Modelleri Getir
     */
    public function get_supported_models() {
        return $this->supported_models;
    }
    
    /**
     * Model Listesini Provider'a Göre Getir
     */
    public function get_models_by_provider($provider) {
        return $this->supported_models[$provider] ?? [];
    }
}
