<?php
/**
 * Link Havuzu Yönetimi
 * Çapa metinleri ve linkleri otomatik ekler
 */

if (!defined('ABSPATH')) exit;

class Mimoza_Link_Manager {
    
    /**
     * Link havuzunu getir
     */
    public function get_all_links() {
        $internal_links = get_option('mimoza_internal_links', []);
        $external_links = get_option('mimoza_external_links', []);
        
        return [
            'internal' => $internal_links,
            'external' => $external_links
        ];
    }
    
    /**
     * Link ekle
     */
    public function add_link($anchor, $url, $type = 'internal') {
        $option_name = 'mimoza_' . $type . '_links';
        $links = get_option($option_name, []);
        
        $links[] = [
            'anchor' => sanitize_text_field($anchor),
            'url' => esc_url_raw($url),
            'added_at' => current_time('mysql')
        ];
        
        update_option($option_name, $links);
        return true;
    }
    
    /**
     * Link sil
     */
    public function delete_link($index, $type = 'internal') {
        $option_name = 'mimoza_' . $type . '_links';
        $links = get_option($option_name, []);
        
        if (isset($links[$index])) {
            unset($links[$index]);
            $links = array_values($links); // Reindex
            update_option($option_name, $links);
            return true;
        }
        
        return false;
    }
    
    /**
     * İçeriğe linkleri ekle
     */
    public function add_links_to_content($content) {
        $links = $this->get_all_links();
        $all_links = array_merge($links['internal'], $links['external']);
        
        if (empty($all_links)) {
            return $content;
        }
        
        // Rastgele 2-5 link ekle
        shuffle($all_links);
        $selected_links = array_slice($all_links, 0, rand(2, 5));
        
        // Her link için bir kez ekle
        $used_anchors = [];
        
        foreach ($selected_links as $link) {
            $anchor = $link['anchor'];
            $url = $link['url'];
            
            // Bu anchor daha önce kullanıldıysa atla
            if (in_array($anchor, $used_anchors)) {
                continue;
            }
            
            // Link türüne göre rel attribute
            $is_internal = strpos($url, home_url()) !== false;
            $rel = $is_internal ? '' : ' rel="nofollow noopener"';
            
            // Anchor metnini bul ve linkle değiştir (case-insensitive, sadece bir kez)
            $pattern = '/\b' . preg_quote($anchor, '/') . '\b/i';
            $replacement = '<a href="' . $url . '"' . $rel . '>' . $anchor . '</a>';
            
            $content = preg_replace($pattern, $replacement, $content, 1);
            $used_anchors[] = $anchor;
        }
        
        return $content;
    }
    
    /**
     * Toplu link import (CSV)
     */
    public function import_links_from_csv($file_path, $type = 'internal') {
        if (!file_exists($file_path)) {
            return new WP_Error('file_not_found', 'Dosya bulunamadı');
        }
        
        $handle = fopen($file_path, 'r');
        $imported = 0;
        
        // İlk satırı atla (başlık)
        fgetcsv($handle);
        
        while (($data = fgetcsv($handle)) !== false) {
            if (count($data) >= 2) {
                $this->add_link($data[0], $data[1], $type);
                $imported++;
            }
        }
        
        fclose($handle);
        
        return $imported;
    }
    
    /**
     * Link havuzunu export et (CSV)
     */
    public function export_links_to_csv($type = 'all') {
        $links = $this->get_all_links();
        
        $filename = 'mimoza-links-' . date('Y-m-d') . '.csv';
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=' . $filename);
        
        $output = fopen('php://output', 'w');
        
        // UTF-8 BOM ekle (Excel için)
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        // Başlık
        fputcsv($output, ['Çapa Metin', 'URL', 'Tip']);
        
        if ($type === 'all' || $type === 'internal') {
            foreach ($links['internal'] as $link) {
                fputcsv($output, [$link['anchor'], $link['url'], 'İç Link']);
            }
        }
        
        if ($type === 'all' || $type === 'external') {
            foreach ($links['external'] as $link) {
                fputcsv($output, [$link['anchor'], $link['url'], 'Dış Link']);
            }
        }
        
        fclose($output);
        exit;
    }
    
    /**
     * AJAX İşlemleri
     */
    public function __construct() {
        add_action('wp_ajax_mimoza_add_link', [$this, 'ajax_add_link']);
        add_action('wp_ajax_mimoza_delete_link', [$this, 'ajax_delete_link']);
        add_action('wp_ajax_mimoza_export_links', [$this, 'ajax_export_links']);
    }
    
    public function ajax_add_link() {
        check_ajax_referer('mimoza_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Yetkiniz yok');
        }
        
        $anchor = sanitize_text_field($_POST['anchor']);
        $url = esc_url_raw($_POST['url']);
        $type = sanitize_text_field($_POST['type']);
        
        $result = $this->add_link($anchor, $url, $type);
        
        if ($result) {
            wp_send_json_success('Link eklendi');
        } else {
            wp_send_json_error('Link eklenemedi');
        }
    }
    
    public function ajax_delete_link() {
        check_ajax_referer('mimoza_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Yetkiniz yok');
        }
        
        $index = intval($_POST['index']);
        $type = sanitize_text_field($_POST['type']);
        
        $result = $this->delete_link($index, $type);
        
        if ($result) {
            wp_send_json_success('Link silindi');
        } else {
            wp_send_json_error('Link silinemedi');
        }
    }
    
    public function ajax_export_links() {
        check_ajax_referer('mimoza_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Yetkiniz yok');
        }
        
        $type = sanitize_text_field($_GET['type'] ?? 'all');
        $this->export_links_to_csv($type);
    }
}