<?php
/**
 * Plugin Name: Mimoza AI SEO
 * Description: Ücretsiz AI İçerik Üretimi - ChatGPT, Claude, Gemini. Sınırsız makale üretimi, gelişmiş zamanlama, SEO optimizasyonu.
 * Version: 5.1-FREE
 * Author: Mimoza Bilişim
 * License: GPL v2 or later
 * Text Domain: mimoza
 */

if (!defined('ABSPATH')) exit;

define('MIMOZA_VERSION', '5.1-FREE');
define('MIMOZA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MIMOZA_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Sınıfları yükle
require_once MIMOZA_PLUGIN_PATH . 'includes/database-setup.php';
require_once MIMOZA_PLUGIN_PATH . 'includes/class-seo-optimizer.php';
require_once MIMOZA_PLUGIN_PATH . 'includes/class-link-manager.php';
require_once MIMOZA_PLUGIN_PATH . 'includes/ajax-handlers.php';

class Mimoza_AI_SEO {

    private $seo_optimizer;
    private $link_manager;

    public function __construct() {
        $this->seo_optimizer = new Mimoza_SEO_Optimizer();
        $this->link_manager = new Mimoza_Link_Manager();

        // Hook'lar
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // Form işleme
        add_action('admin_post_mimoza_bulk_generate', [$this, 'handle_admin_post_bulk_generate']);
        
        // Aktivasyon
        register_activation_hook(__FILE__, [$this, 'activate']);
    }

    public function add_admin_menu() {
        add_menu_page(
            'Mimoza AI SEO',
            'Mimoza AI SEO',
            'manage_options',
            'mimoza-ai-seo',
            [$this, 'render_dashboard'],
            'dashicons-edit-page',
            30
        );
        
        add_submenu_page('mimoza-ai-seo', 'İçerik Üretici', 'İçerik Üretici', 'manage_options', 'mimoza-content-generator', [$this, 'render_content_generator']);
        add_submenu_page('mimoza-ai-seo', 'Link Havuzu', 'Link Havuzu', 'manage_options', 'mimoza-link-pool', [$this, 'render_link_pool']);
        add_submenu_page('mimoza-ai-seo', 'Zamanlama', 'Zamanlama', 'manage_options', 'mimoza-scheduling', [$this, 'render_scheduling']);
        add_submenu_page('mimoza-ai-seo', 'Ayarlar', 'Ayarlar', 'manage_options', 'mimoza-settings', [$this, 'render_settings']);
    }

    public function render_dashboard() {
        $stats = $this->get_dashboard_stats();
        include MIMOZA_PLUGIN_PATH . 'admin/dashboard.php';
    }

    public function render_content_generator() {
        include MIMOZA_PLUGIN_PATH . 'admin/content-generator.php';
    }

    public function render_link_pool() {
        $links = $this->link_manager->get_all_links();
        include MIMOZA_PLUGIN_PATH . 'admin/link-pool.php';
    }

    public function render_scheduling() {
        include MIMOZA_PLUGIN_PATH . 'admin/scheduling.php';
    }

    public function render_settings() {
        include MIMOZA_PLUGIN_PATH . 'admin/settings.php';
    }

    public function handle_admin_post_bulk_generate() {
        error_log('MIMOZA: handle_admin_post_bulk_generate called');
        
        // Yetki kontrolü
        if (!current_user_can('manage_options')) {
            wp_die('Yetkiniz yok!');
        }
        
        // Nonce kontrolü
        if (!isset($_POST['mimoza_bulk_generate_nonce']) || 
            !wp_verify_nonce($_POST['mimoza_bulk_generate_nonce'], 'mimoza_bulk_generate_action')) {
            error_log('MIMOZA: Nonce check failed');
            wp_die('Güvenlik kontrolü başarısız!');
        }
        
        // Anahtar kelimeleri al
        $keywords = array_filter(array_map('trim', explode("\n", $_POST['keywords'])));
        
        if (empty($keywords)) {
            wp_redirect(admin_url('admin.php?page=mimoza-content-generator&error=' . urlencode('Anahtar kelime girilmedi')));
            exit;
        }
        
        // Ayarları al
        $settings = [
            'model' => sanitize_text_field($_POST['ai_model']),
            'word_count' => intval($_POST['word_count']),
            'publish_option' => sanitize_text_field($_POST['publish_option']),
            'use_images' => isset($_POST['use_images']),
            'image_provider' => sanitize_text_field($_POST['image_provider'] ?? 'pexels'),
            'categories' => isset($_POST['categories']) ? array_map('intval', $_POST['categories']) : [],
            'custom_prompt' => isset($_POST['custom_prompt']) ? sanitize_textarea_field($_POST['custom_prompt']) : '',
        ];
        
        error_log('MIMOZA: Processing ' . count($keywords) . ' keywords');
        if (!empty($settings['custom_prompt'])) {
            error_log('MIMOZA: Using custom prompt');
        }
        
        // Üretim başlat
        $generated = 0;
        $errors = [];
        
        foreach ($keywords as $keyword) {
            error_log('MIMOZA: Processing keyword: ' . $keyword);
            
            try {
                // AI ile içerik üret
                $content = $this->generate_content_with_ai($keyword, $settings);
                
                if (is_wp_error($content)) {
                    $errors[] = $keyword . ': ' . $content->get_error_message();
                    error_log('MIMOZA: Error: ' . $content->get_error_message());
                    continue;
                }
                
                // Post oluştur
                $post_data = [
                    'post_title' => $keyword,
                    'post_content' => $content,
                    'post_status' => $settings['publish_option'],
                    'post_author' => get_current_user_id(),
                    'post_category' => $settings['categories']
                ];
                
                $post_id = wp_insert_post($post_data);
                
                if (is_wp_error($post_id)) {
                    $errors[] = $keyword . ': ' . $post_id->get_error_message();
                    continue;
                }
                
                error_log('MIMOZA: Post created with ID: ' . $post_id);
                
                // SEO meta
                $seo_data = $this->seo_optimizer->optimize_for_keyword($keyword, $content);
                $this->seo_optimizer->save_seo_meta($post_id, $seo_data);
                
                // Resim ekle
                if ($settings['use_images']) {
                    $this->add_featured_image($post_id, $keyword, $settings['image_provider']);
                }
                
                // Kullanım logla
                global $wpdb;
                $wpdb->insert($wpdb->prefix . 'mimoza_generations', [
                    'post_id' => $post_id,
                    'keyword' => $keyword,
                    'model' => $settings['model'],
                    'word_count' => $settings['word_count'],
                    'created_at' => current_time('mysql')
                ]);
                
                $generated++;
                
            } catch (Exception $e) {
                $errors[] = $keyword . ': ' . $e->getMessage();
                error_log('MIMOZA: Exception: ' . $e->getMessage());
            }
        }
        
        error_log('MIMOZA: Completed. Generated: ' . $generated);
        
        // Redirect
        if ($generated > 0) {
            wp_redirect(admin_url('admin.php?page=mimoza-content-generator&generated=1&count=' . $generated));
        } else {
            $error_msg = !empty($errors) ? implode(', ', array_slice($errors, 0, 3)) : 'Bilinmeyen hata';
            wp_redirect(admin_url('admin.php?page=mimoza-content-generator&error=' . urlencode($error_msg)));
        }
        exit;
    }

    private function generate_content_with_ai($keyword, $settings) {
        $model = $settings['model'];
        $word_count = $settings['word_count'];
        $custom_prompt = $settings['custom_prompt'] ?? '';
        
        $prompt = $this->create_optimized_prompt($keyword, $word_count, $custom_prompt);
        
        error_log('MIMOZA: Calling AI with model: ' . $model);
        
        // Model'e göre API çağrısı
        if (strpos($model, 'gpt') !== false) {
            return $this->call_openai($prompt, $model);
        } elseif (strpos($model, 'claude') !== false) {
            return $this->call_claude($prompt, $model);
        } elseif (strpos($model, 'gemini') !== false) {
            return $this->call_gemini($prompt, $model);
        }
        
        return new WP_Error('invalid_model', 'Geçersiz AI modeli');
    }

    private function create_optimized_prompt($keyword, $word_count, $custom_prompt = '') {
        // Eğer özel prompt varsa, onu kullan
        if (!empty($custom_prompt)) {
            // {keyword} placeholder'ını değiştir
            $prompt = str_replace('{keyword}', $keyword, $custom_prompt);
            
            // Kelime sayısını da belirtmek için ekle
            $prompt .= "\n\nNOT: Yaklaşık {$word_count} kelime uzunluğunda yaz.";
            
            return $prompt;
        }
        
        // Varsayılan SEO uyumlu prompt
        return "TÜRKÇE profesyonel bir SEO makalesi yaz.

KONU: {$keyword}

KURALLAR:
- Tam {$word_count} kelime yaz
- H2, H3, H4 başlıkları kullan (H1 kullanma)
- Her paragraf maksimum 3-4 cümle olsun
- Cümleler kısa ve anlaşılır olsun
- Doğal ve akıcı dil kullan
- Anahtar kelimeyi doğal şekilde yerleştir
- Okuyucuya hitap et (sen dili kullan)
- Giriş, gelişme, sonuç yapısında ol

FORMAT:
<h2>Ana Başlık 1</h2>
<p>Paragraf metni...</p>

<h3>Alt Başlık</h3>
<p>Paragraf metni...</p>

BAŞLA:";
    }

    private function call_openai($prompt, $model) {
        $api_key = get_option('mimoza_openai_api_key');
        
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'OpenAI API anahtarı girilmemiş');
        }
        
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode([
                'model' => $model,
                'messages' => [['role' => 'user', 'content' => $prompt]],
                'max_tokens' => 4000,
                'temperature' => 0.7
            ]),
            'timeout' => 120
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            $body = wp_remote_retrieve_body($response);
            return new WP_Error('api_error', 'OpenAI API error: ' . $body);
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['choices'][0]['message']['content'] ?? '';
    }

    private function call_claude($prompt, $model) {
        $api_key = get_option('mimoza_claude_api_key');
        
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'Claude API anahtarı girilmemiş');
        }
        
        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'headers' => [
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01',
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode([
                'model' => $model,
                'max_tokens' => 4000,
                'messages' => [['role' => 'user', 'content' => $prompt]]
            ]),
            'timeout' => 120
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            $body = wp_remote_retrieve_body($response);
            return new WP_Error('api_error', 'Claude API error: ' . $body);
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['content'][0]['text'] ?? '';
    }

    private function call_gemini($prompt, $model) {
        $api_key = get_option('mimoza_gemini_api_key');
        
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'Gemini API anahtarı girilmemiş');
        }
        
        error_log('MIMOZA: Calling Gemini API');
        
        $response = wp_remote_post(
            'https://generativelanguage.googleapis.com/v1beta/models/' . $model . ':generateContent?key=' . $api_key,
            [
                'headers' => ['Content-Type' => 'application/json'],
                'body' => json_encode([
                    'contents' => [['parts' => [['text' => $prompt]]]],
                    'generationConfig' => ['maxOutputTokens' => 8000]
                ]),
                'timeout' => 120
            ]
        );

        if (is_wp_error($response)) {
            error_log('MIMOZA: Gemini WP_Error: ' . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        error_log('MIMOZA: Gemini response code: ' . $code);
        
        if ($code !== 200) {
            error_log('MIMOZA: Gemini error body: ' . $body);
            return new WP_Error('api_error', 'Gemini API error (Code ' . $code . '): ' . $body);
        }

        $data = json_decode($body, true);
        
        if (isset($data['candidates'][0]['content']['parts'][0]['text'])) {
            $content = $data['candidates'][0]['content']['parts'][0]['text'];
            error_log('MIMOZA: Content generated successfully, length: ' . strlen($content));
            return $content;
        }
        
        error_log('MIMOZA: Gemini unexpected response: ' . print_r($data, true));
        return new WP_Error('no_content', 'Gemini yanıt verdi ama içerik boş');
    }

    private function add_featured_image($post_id, $keyword, $provider) {
        $image_url = $this->search_image($keyword, $provider);
        
        if ($image_url) {
            $this->set_featured_image_from_url($post_id, $image_url, $keyword);
        }
    }

    private function search_image($keyword, $provider) {
        switch ($provider) {
            case 'pexels':
                return $this->search_pexels($keyword);
            case 'unsplash':
                return $this->search_unsplash($keyword);
            case 'pixabay':
                return $this->search_pixabay($keyword);
            default:
                return false;
        }
    }

    private function search_pexels($keyword) {
        $api_key = get_option('mimoza_pexels_api_key');
        if (empty($api_key)) return false;
        
        $response = wp_remote_get(
            'https://api.pexels.com/v1/search?query=' . urlencode($keyword) . '&per_page=1',
            ['headers' => ['Authorization' => $api_key]]
        );
        
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            return $data['photos'][0]['src']['large'] ?? false;
        }
        
        return false;
    }

    private function search_unsplash($keyword) {
        $api_key = get_option('mimoza_unsplash_api_key');
        if (empty($api_key)) return false;
        
        $response = wp_remote_get(
            'https://api.unsplash.com/search/photos?query=' . urlencode($keyword) . '&per_page=1',
            ['headers' => ['Authorization' => 'Client-ID ' . $api_key]]
        );
        
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            return $data['results'][0]['urls']['regular'] ?? false;
        }
        
        return false;
    }

    private function search_pixabay($keyword) {
        $api_key = get_option('mimoza_pixabay_api_key');
        if (empty($api_key)) return false;
        
        $response = wp_remote_get(
            'https://pixabay.com/api/?key=' . $api_key . '&q=' . urlencode($keyword) . '&per_page=1'
        );
        
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            return $data['hits'][0]['largeImageURL'] ?? false;
        }
        
        return false;
    }

    private function set_featured_image_from_url($post_id, $image_url, $alt_text) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        $tmp = download_url($image_url);
        
        if (is_wp_error($tmp)) {
            return false;
        }
        
        $file_array = [
            'name' => sanitize_file_name($alt_text) . '-' . time() . '.jpg',
            'tmp_name' => $tmp
        ];
        
        $id = media_handle_sideload($file_array, $post_id);
        
        if (!is_wp_error($id)) {
            set_post_thumbnail($post_id, $id);
            update_post_meta($id, '_wp_attachment_image_alt', $alt_text);
            return true;
        }
        
        @unlink($tmp);
        return false;
    }

    private function get_dashboard_stats() {
        global $wpdb;
        
        $table = $wpdb->prefix . 'mimoza_generations';
        
        return [
            'today' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE DATE(created_at) = CURDATE()"),
            'week' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)"),
            'month' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)"),
            'total' => $wpdb->get_var("SELECT COUNT(*) FROM $table")
        ];
    }

    public function register_settings() {
        register_setting('mimoza_settings', 'mimoza_openai_api_key');
        register_setting('mimoza_settings', 'mimoza_claude_api_key');
        register_setting('mimoza_settings', 'mimoza_gemini_api_key');
        register_setting('mimoza_settings', 'mimoza_pexels_api_key');
        register_setting('mimoza_settings', 'mimoza_unsplash_api_key');
        register_setting('mimoza_settings', 'mimoza_pixabay_api_key');
    }

    public function enqueue_assets($hook) {
        if (strpos($hook, 'mimoza') === false) return;
        
        wp_enqueue_style('mimoza-admin', MIMOZA_PLUGIN_URL . 'assets/css/admin.css', [], MIMOZA_VERSION);
        wp_enqueue_script('mimoza-admin', MIMOZA_PLUGIN_URL . 'assets/js/admin.js', ['jquery'], MIMOZA_VERSION, true);
        
        wp_localize_script('mimoza-admin', 'mimoza', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mimoza_nonce')
        ]);
    }

    public function activate() {
        Mimoza_Database_Setup::install();
    }
}

new Mimoza_AI_SEO();




// PayTR ve Satış Sistemi
if (class_exists('WooCommerce')) {
    require_once MIMOZA_PLUGIN_PATH . 'includes/class-paytr-gateway.php';
    require_once MIMOZA_PLUGIN_PATH . 'includes/class-license-shop.php';
    require_once MIMOZA_PLUGIN_PATH . 'includes/class-upgrade-widget.php';
}
