/**
 * Mimoza AI SEO - Admin JavaScript (Free Version)
 */

(function($) {
    'use strict';
    
    // Doküman hazır olduğunda
    $(document).ready(function() {
        
        // Form doğrulama
        $('#mimoza-content-form').on('submit', function(e) {
            const keywords = $('textarea[name="keywords"]').val().trim();
            
            if (!keywords) {
                e.preventDefault();
                alert('Lütfen en az bir anahtar kelime girin!');
                return false;
            }
            
            const keywordCount = keywords.split('\n').filter(k => k.trim()).length;
            
            if (!confirm('📝 ' + keywordCount + ' makale oluşturulacak.\n\n⏱️ Tahmini süre: ' + Math.ceil(keywordCount * 0.5) + ' dakika\n\nDevam etmek istiyor musunuz?')) {
                e.preventDefault();
                return false;
            }
            
            // Loading state
            const $button = $(this).find('button[type="submit"]');
            $button.prop('disabled', true).html('⏳ Oluşturuluyor... (' + keywordCount + ' makale)');
        });
        
        // Link ekleme formu - İç linkler
        $('#add-internal-link').on('submit', function(e) {
            e.preventDefault();
            addLink('internal', $(this));
        });
        
        // Link ekleme formu - Dış linkler
        $('#add-external-link').on('submit', function(e) {
            e.preventDefault();
            addLink('external', $(this));
        });
        
        // Link silme
        $('.delete-link').on('click', function() {
            if (!confirm('Bu linki silmek istediğinize emin misiniz?')) {
                return;
            }
            
            const index = $(this).data('index');
            const type = $(this).data('type');
            
            deleteLink(index, type);
        });
        
        // Tab değiştirme
        $('.tab-button').on('click', function() {
            const targetTab = $(this).data('tab');
            
            $('.tab-button').removeClass('active');
            $(this).addClass('active');
            
            $('.tab-content').removeClass('active');
            $(`#${targetTab}-links, #import-export`).addClass('active');
        });
        
        // Şifre göster/gizle
        $('.toggle-password').on('click', function() {
            const $input = $(this).prev('input');
            const type = $input.attr('type');
            
            if (type === 'password') {
                $input.attr('type', 'text');
                $(this).text('🙈');
            } else {
                $input.attr('type', 'password');
                $(this).text('👁️');
            }
        });
        
        // CSV içe aktarma
        $('#import-csv-form').on('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'mimoza_import_csv');
            formData.append('nonce', mimoza.nonce);
            
            $.ajax({
                url: mimoza.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        showNotification('success', response.data.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        showNotification('error', response.data);
                    }
                },
                error: function() {
                    showNotification('error', 'Bir hata oluştu!');
                }
            });
        });
        
        // Anahtar kelime sayacı
        $('textarea[name="keywords"]').on('input', function() {
            const keywords = $(this).val().split('\n').filter(k => k.trim());
            const count = keywords.length;
            
            let counterHtml = `<span style="color: green;">${count}</span> anahtar kelime`;
            
            if (!$('.keyword-counter').length) {
                $(this).after(`<p class="description keyword-counter">${counterHtml}</p>`);
            } else {
                $('.keyword-counter').html(counterHtml);
            }
        });
        
        // Kelime sayısı display
        $('input[name="word_count"]').on('input', function() {
            const value = $(this).val();
            if (!$('.word-count-display').length) {
                $(this).after(`<span class="word-count-display" style="margin-left: 10px; font-weight: bold;">${value} kelime</span>`);
            } else {
                $('.word-count-display').text(`${value} kelime`);
            }
        });
        
        // Otomatik kaydetme uyarısı
        let formChanged = false;
        $('form input, form textarea, form select').on('change', function() {
            formChanged = true;
        });
        
        $(window).on('beforeunload', function() {
            if (formChanged) {
                return 'Kaydedilmemiş değişiklikler var. Sayfadan ayrılmak istediğinize emin misiniz?';
            }
        });
        
        $('form').on('submit', function() {
            formChanged = false;
        });
    });
    
    /**
     * Link ekleme fonksiyonu
     */
    function addLink(type, $form) {
        const data = {
            action: 'mimoza_add_link',
            nonce: mimoza.nonce,
            type: type,
            anchor: $form.find('input[name="anchor"]').val(),
            url: $form.find('input[name="url"]').val()
        };
        
        $.post(mimoza.ajax_url, data, function(response) {
            if (response.success) {
                showNotification('success', 'Link eklendi!');
                setTimeout(() => location.reload(), 1000);
            } else {
                showNotification('error', response.data);
            }
        });
    }
    
    /**
     * Link silme fonksiyonu
     */
    function deleteLink(index, type) {
        $.post(mimoza.ajax_url, {
            action: 'mimoza_delete_link',
            nonce: mimoza.nonce,
            index: index,
            type: type
        }, function(response) {
            if (response.success) {
                showNotification('success', 'Link silindi!');
                setTimeout(() => location.reload(), 1000);
            } else {
                showNotification('error', response.data);
            }
        });
    }
    
    /**
     * Bildirim göster
     */
    function showNotification(type, message) {
        const icons = {
            success: '✔',
            error: '✗',
            warning: '⚠',
            info: 'ℹ'
        };
        
        const colors = {
            success: '#46b450',
            error: '#dc3232',
            warning: '#ffb900',
            info: '#00a0d2'
        };
        
        const $notification = $('<div>', {
            class: 'mimoza-notification',
            css: {
                position: 'fixed',
                top: '50px',
                right: '20px',
                background: 'white',
                padding: '20px 25px',
                borderRadius: '10px',
                boxShadow: '0 5px 20px rgba(0,0,0,0.2)',
                zIndex: 99999,
                minWidth: '300px',
                borderLeft: `4px solid ${colors[type]}`,
                animation: 'slideInRight 0.3s ease'
            },
            html: `
                <div style="display: flex; align-items: center; gap: 15px;">
                    <span style="font-size: 24px; color: ${colors[type]};">${icons[type]}</span>
                    <span style="flex: 1;">${message}</span>
                    <button class="notification-close" style="background: none; border: none; font-size: 20px; cursor: pointer; color: #999;">×</button>
                </div>
            `
        });
        
        $('body').append($notification);
        
        $notification.find('.notification-close').on('click', function() {
            $notification.fadeOut(300, function() {
                $(this).remove();
            });
        });
        
        setTimeout(() => {
            $notification.fadeOut(300, function() {
                $(this).remove();
            });
        }, 5000);
    }
    
})(jQuery);

// CSS animasyonları ekle
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(400px);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
`;
document.head.appendChild(style);