<?php
/**
 * SEO Optimizasyon Sınıfı
 * Page Title ve Meta Description otomatik üretimi
 */

if (!defined('ABSPATH')) exit;

class Mimoza_SEO_Optimizer {
    
    public function optimize_for_keyword($keyword, $content) {
        return [
            'title' => $this->generate_seo_title($keyword),
            'description' => $this->generate_meta_description($keyword, $content),
            'focus_keyword' => $keyword
        ];
    }
    
    /**
     * SEO Title Üret (Max 55 karakter)
     */
    public function generate_seo_title($keyword) {
        $suffixes = [
            ' - Detaylı Rehber',
            ' - Bilmeniz Gerekenler',
            ' - Kapsamlı İnceleme',
            ' - Uzman Önerileri',
            ' - 2024 Rehberi',
            ' Hakkında Her Şey',
            ' - En İyi Yöntemler'
        ];
        
        // Rastgele bir suffix seç
        $suffix = $suffixes[array_rand($suffixes)];
        $title = $keyword . $suffix;
        
        // 55 karakteri aşıyorsa kısalt
        if (mb_strlen($title) > 55) {
            $title = mb_substr($keyword, 0, 50) . '...';
        }
        
        return $title;
    }
    
    /**
     * Meta Description Üret (Max 130 karakter)
     */
    public function generate_meta_description($keyword, $content) {
        // İçeriğin ilk paragrafını al
        $text = wp_strip_all_tags($content);
        
        // İlk cümleyi bul
        preg_match('/^.+?[.!?]/', $text, $matches);
        $first_sentence = $matches[0] ?? wp_trim_words($text, 15);
        
        // Anahtar kelime ile birleştir
        $description = $keyword . ' hakkında: ' . $first_sentence;
        
        // 130 karakteri aşıyorsa kısalt
        if (mb_strlen($description) > 130) {
            $description = mb_substr($description, 0, 127) . '...';
        }
        
        return $description;
    }
    
    /**
     * SEO Meta Verilerini Kaydet (Yoast & Rank Math)
     */
    public function save_seo_meta($post_id, $seo_data) {
        $title = $seo_data['title'];
        $description = $seo_data['description'];
        $focus_keyword = $seo_data['focus_keyword'];
        
        // Yoast SEO
        if (defined('WPSEO_VERSION')) {
            update_post_meta($post_id, '_yoast_wpseo_title', $title);
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $description);
            update_post_meta($post_id, '_yoast_wpseo_focuskw', $focus_keyword);
        }
        
        // Rank Math
        if (class_exists('RankMath')) {
            update_post_meta($post_id, 'rank_math_title', $title);
            update_post_meta($post_id, 'rank_math_description', $description);
            update_post_meta($post_id, 'rank_math_focus_keyword', $focus_keyword);
        }
        
        // All in One SEO
        if (class_exists('AIOSEO')) {
            update_post_meta($post_id, '_aioseo_title', $title);
            update_post_meta($post_id, '_aioseo_description', $description);
        }
        
        // Mimoza kendi meta'sı
        update_post_meta($post_id, '_mimoza_seo_title', $title);
        update_post_meta($post_id, '_mimoza_seo_description', $description);
        update_post_meta($post_id, '_mimoza_focus_keyword', $focus_keyword);
        
        return true;
    }
    
    /**
     * İçeriğe H2, H3, H4 ekle
     */
    public function add_headings_to_content($content) {
        // İçerik zaten HTML başlıkları içeriyorsa dokunma
        if (preg_match('/<h[2-4]>/i', $content)) {
            return $content;
        }
        
        // Paragrafları böl
        $paragraphs = explode("\n\n", $content);
        $structured = '';
        
        foreach ($paragraphs as $index => $paragraph) {
            if (empty(trim($paragraph))) continue;
            
            // Her 3-4 paragrafta bir H2 ekle
            if ($index % 4 === 0 && $index > 0) {
                $heading = $this->extract_heading_from_paragraph($paragraph);
                $structured .= "<h2>{$heading}</h2>\n";
            } elseif ($index % 2 === 0 && $index > 0) {
                $heading = $this->extract_heading_from_paragraph($paragraph);
                $structured .= "<h3>{$heading}</h3>\n";
            }
            
            $structured .= "<p>{$paragraph}</p>\n\n";
        }
        
        return $structured;
    }
    
    private function extract_heading_from_paragraph($paragraph) {
        // İlk cümleyi başlık yap
        $sentences = preg_split('/[.!?]/', $paragraph);
        $heading = trim($sentences[0]);
        
        // Çok uzunsa kısalt
        if (mb_strlen($heading) > 60) {
            $heading = mb_substr($heading, 0, 57) . '...';
        }
        
        return $heading;
    }
}