<?php
/**
 * Plugin Name: Mimoza AI SEO Premium v6.0
 * Description: Gelişmiş AI İçerik Üretimi - ChatGPT-5, Claude Sonnet 4.5, Gemini 2.0, Jasper. Otomatik linking, SEO optimizasyonu, akıllı prompt sistemi.
 * Version: 6.0
 * Author: Mimoza Bilişim
 * License: GPL v2 or later
 * Text Domain: mimoza
 */

if (!defined('ABSPATH')) exit;

define('MIMOZA_VERSION', '6.0');
define('MIMOZA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MIMOZA_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Sınıfları yükle
require_once MIMOZA_PLUGIN_PATH . 'includes/database-setup.php';
require_once MIMOZA_PLUGIN_PATH . 'includes/class-seo-optimizer.php';
require_once MIMOZA_PLUGIN_PATH . 'includes/class-link-manager.php';
require_once MIMOZA_PLUGIN_PATH . 'includes/class-advanced-ai-engine.php';
require_once MIMOZA_PLUGIN_PATH . 'includes/ajax-handlers.php';

class Mimoza_AI_SEO_Premium {

    private $seo_optimizer;
    private $link_manager;
    private $ai_engine;

    public function __construct() {
        $this->seo_optimizer = new Mimoza_SEO_Optimizer();
        $this->link_manager = new Mimoza_Link_Manager();
        $this->ai_engine = new Mimoza_Advanced_AI_Engine();

        // Hook'lar
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // Form işleme
        add_action('admin_post_mimoza_bulk_generate', [$this, 'handle_admin_post_bulk_generate']);
        
        // Aktivasyon
        register_activation_hook(__FILE__, [$this, 'activate']);
        
        // Admin notice
        add_action('admin_notices', [$this, 'show_admin_notices']);
    }

    public function add_admin_menu() {
        add_menu_page(
            'Mimoza AI SEO Premium',
            'Mimoza AI SEO',
            'manage_options',
            'mimoza-ai-seo',
            [$this, 'render_dashboard'],
            'dashicons-edit-page',
            30
        );
        
        add_submenu_page('mimoza-ai-seo', 'İçerik Üretici', 'İçerik Üretici', 'manage_options', 'mimoza-content-generator', [$this, 'render_content_generator']);
        add_submenu_page('mimoza-ai-seo', 'Link Havuzu', 'Link Havuzu', 'manage_options', 'mimoza-link-pool', [$this, 'render_link_pool']);
        add_submenu_page('mimoza-ai-seo', 'Zamanlama', 'Zamanlama', 'manage_options', 'mimoza-scheduling', [$this, 'render_scheduling']);
        add_submenu_page('mimoza-ai-seo', 'Ayarlar', 'Ayarlar', 'manage_options', 'mimoza-settings', [$this, 'render_settings']);
    }

    public function render_dashboard() {
        $stats = $this->get_dashboard_stats();
        include MIMOZA_PLUGIN_PATH . 'admin/dashboard.php';
    }

    public function render_content_generator() {
        $models = $this->ai_engine->get_supported_models();
        include MIMOZA_PLUGIN_PATH . 'admin/content-generator-v6.php';
    }

    public function render_link_pool() {
        $links = $this->link_manager->get_all_links();
        include MIMOZA_PLUGIN_PATH . 'admin/link-pool.php';
    }

    public function render_scheduling() {
        include MIMOZA_PLUGIN_PATH . 'admin/scheduling.php';
    }

    public function render_settings() {
        include MIMOZA_PLUGIN_PATH . 'admin/settings-v6.php';
    }

    /**
     * Gelişmiş Toplu İçerik Üretimi
     */
    public function handle_admin_post_bulk_generate() {
        error_log('MIMOZA v6.0: Bulk generation started');
        
        // Yetki kontrolü
        if (!current_user_can('manage_options')) {
            wp_die('Yetkiniz yok!');
        }
        
        // Nonce kontrolü
        if (!isset($_POST['mimoza_bulk_generate_nonce']) || 
            !wp_verify_nonce($_POST['mimoza_bulk_generate_nonce'], 'mimoza_bulk_generate_action')) {
            error_log('MIMOZA v6.0: Nonce check failed');
            wp_die('Güvenlik kontrolü başarısız!');
        }
        
        // Anahtar kelimeleri al
        $keywords = array_filter(array_map('trim', explode("\n", $_POST['keywords'])));
        
        if (empty($keywords)) {
            wp_redirect(admin_url('admin.php?page=mimoza-content-generator&error=' . urlencode('Anahtar kelime girilmedi')));
            exit;
        }
        
        // Ayarları al
        $settings = [
            'model' => sanitize_text_field($_POST['ai_model']),
            'word_count' => intval($_POST['word_count']),
            'publish_option' => sanitize_text_field($_POST['publish_option']),
            'use_images' => isset($_POST['use_images']),
            'image_provider' => sanitize_text_field($_POST['image_provider'] ?? 'pexels'),
            'categories' => isset($_POST['categories']) ? array_map('intval', $_POST['categories']) : [],
            'custom_prompt' => isset($_POST['custom_prompt']) ? sanitize_textarea_field($_POST['custom_prompt']) : '',
            'language' => sanitize_text_field($_POST['language'] ?? 'tr'),
            'add_internal_links' => isset($_POST['add_internal_links']),
            'add_external_links' => isset($_POST['add_external_links']),
            'auto_seo' => isset($_POST['auto_seo'])
        ];
        
        error_log('MIMOZA v6.0: Processing ' . count($keywords) . ' keywords with model: ' . $settings['model']);
        
        // Üretim başlat
        $generated = 0;
        $errors = [];
        
        foreach ($keywords as $keyword) {
            error_log('MIMOZA v6.0: Processing keyword: ' . $keyword);
            
            try {
                // Gelişmiş AI ile içerik üret
                $content = $this->ai_engine->generate_content($keyword, $settings);
                
                if (is_wp_error($content)) {
                    $errors[] = $keyword . ': ' . $content->get_error_message();
                    error_log('MIMOZA v6.0: Error: ' . $content->get_error_message());
                    continue;
                }
                
                // SEO başlık ve açıklama oluştur
                $seo_title = $this->seo_optimizer->generate_seo_title($keyword);
                $seo_description = $this->seo_optimizer->generate_meta_description($keyword, $content);
                
                // Post oluştur
                $post_data = [
                    'post_title' => $seo_title,
                    'post_content' => $content,
                    'post_status' => $settings['publish_option'],
                    'post_author' => get_current_user_id(),
                    'post_category' => $settings['categories']
                ];
                
                $post_id = wp_insert_post($post_data);
                
                if (is_wp_error($post_id)) {
                    $errors[] = $keyword . ': ' . $post_id->get_error_message();
                    continue;
                }
                
                error_log('MIMOZA v6.0: Post created with ID: ' . $post_id);
                
                // SEO meta kaydet
                if ($settings['auto_seo']) {
                    $seo_data = [
                        'title' => $seo_title,
                        'description' => $seo_description,
                        'focus_keyword' => $keyword
                    ];
                    $this->seo_optimizer->save_seo_meta($post_id, $seo_data);
                }
                
                // Resim ekle
                if ($settings['use_images']) {
                    $this->add_featured_image($post_id, $keyword, $settings['image_provider']);
                }
                
                // Etiketler ekle
                wp_set_post_tags($post_id, [$keyword], true);
                
                // Kullanım logla
                global $wpdb;
                $wpdb->insert($wpdb->prefix . 'mimoza_generations', [
                    'post_id' => $post_id,
                    'keyword' => $keyword,
                    'model' => $settings['model'],
                    'word_count' => $settings['word_count'],
                    'created_at' => current_time('mysql')
                ]);
                
                $generated++;
                
            } catch (Exception $e) {
                $errors[] = $keyword . ': ' . $e->getMessage();
                error_log('MIMOZA v6.0: Exception: ' . $e->getMessage());
            }
        }
        
        error_log('MIMOZA v6.0: Completed. Generated: ' . $generated);
        
        // Redirect
        if ($generated > 0) {
            wp_redirect(admin_url('admin.php?page=mimoza-content-generator&generated=1&count=' . $generated));
        } else {
            $error_msg = !empty($errors) ? implode(', ', array_slice($errors, 0, 3)) : 'Bilinmeyen hata';
            wp_redirect(admin_url('admin.php?page=mimoza-content-generator&error=' . urlencode($error_msg)));
        }
        exit;
    }

    /**
     * Öne Çıkan Resim Ekle
     */
    private function add_featured_image($post_id, $keyword, $provider) {
        $image_url = $this->search_image($keyword, $provider);
        
        if ($image_url) {
            $this->set_featured_image_from_url($post_id, $image_url, $keyword);
        }
    }

    private function search_image($keyword, $provider) {
        switch ($provider) {
            case 'pexels':
                return $this->search_pexels($keyword);
            case 'unsplash':
                return $this->search_unsplash($keyword);
            case 'pixabay':
                return $this->search_pixabay($keyword);
            default:
                return false;
        }
    }

    private function search_pexels($keyword) {
        $api_key = get_option('mimoza_pexels_api_key');
        if (empty($api_key)) return false;
        
        $response = wp_remote_get(
            'https://api.pexels.com/v1/search?query=' . urlencode($keyword) . '&per_page=1',
            ['headers' => ['Authorization' => $api_key]]
        );
        
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            return $data['photos'][0]['src']['large'] ?? false;
        }
        
        return false;
    }

    private function search_unsplash($keyword) {
        $api_key = get_option('mimoza_unsplash_api_key');
        if (empty($api_key)) return false;
        
        $response = wp_remote_get(
            'https://api.unsplash.com/search/photos?query=' . urlencode($keyword) . '&per_page=1',
            ['headers' => ['Authorization' => 'Client-ID ' . $api_key]]
        );
        
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            return $data['results'][0]['urls']['regular'] ?? false;
        }
        
        return false;
    }

    private function search_pixabay($keyword) {
        $api_key = get_option('mimoza_pixabay_api_key');
        if (empty($api_key)) return false;
        
        $response = wp_remote_get(
            'https://pixabay.com/api/?key=' . $api_key . '&q=' . urlencode($keyword) . '&per_page=1'
        );
        
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            return $data['hits'][0]['largeImageURL'] ?? false;
        }
        
        return false;
    }

    private function set_featured_image_from_url($post_id, $image_url, $alt_text) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        $tmp = download_url($image_url);
        
        if (is_wp_error($tmp)) {
            return false;
        }
        
        $file_array = [
            'name' => sanitize_file_name($alt_text) . '-' . time() . '.jpg',
            'tmp_name' => $tmp
        ];
        
        $id = media_handle_sideload($file_array, $post_id);
        
        if (!is_wp_error($id)) {
            set_post_thumbnail($post_id, $id);
            update_post_meta($id, '_wp_attachment_image_alt', $alt_text);
            return true;
        }
        
        @unlink($tmp);
        return false;
    }

    /**
     * Dashboard İstatistikleri
     */
    private function get_dashboard_stats() {
        global $wpdb;
        
        $table = $wpdb->prefix . 'mimoza_generations';
        
        return [
            'today' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE DATE(created_at) = CURDATE()"),
            'week' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)"),
            'month' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)"),
            'total' => $wpdb->get_var("SELECT COUNT(*) FROM $table"),
            'models_used' => $wpdb->get_results("SELECT model, COUNT(*) as count FROM $table GROUP BY model ORDER BY count DESC LIMIT 5")
        ];
    }

    /**
     * Ayarları Kaydet
     */
    public function register_settings() {
        // AI API Keys
        register_setting('mimoza_settings', 'mimoza_openai_api_key');
        register_setting('mimoza_settings', 'mimoza_claude_api_key');
        register_setting('mimoza_settings', 'mimoza_gemini_api_key');
        register_setting('mimoza_settings', 'mimoza_jasper_api_key');
        
        // Image API Keys
        register_setting('mimoza_settings', 'mimoza_pexels_api_key');
        register_setting('mimoza_settings', 'mimoza_unsplash_api_key');
        register_setting('mimoza_settings', 'mimoza_pixabay_api_key');
        
        // Advanced Settings
        register_setting('mimoza_settings', 'mimoza_default_model');
        register_setting('mimoza_settings', 'mimoza_default_word_count');
        register_setting('mimoza_settings', 'mimoza_auto_internal_links');
        register_setting('mimoza_settings', 'mimoza_auto_external_links');
    }

    /**
     * Assets Yükle
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'mimoza') === false) return;
        
        wp_enqueue_style('mimoza-admin', MIMOZA_PLUGIN_URL . 'assets/css/admin.css', [], MIMOZA_VERSION);
        wp_enqueue_script('mimoza-admin', MIMOZA_PLUGIN_URL . 'assets/js/admin.js', ['jquery'], MIMOZA_VERSION, true);
        
        wp_localize_script('mimoza-admin', 'mimoza', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mimoza_nonce')
        ]);
    }

    /**
     * Aktivasyon
     */
    public function activate() {
        Mimoza_Database_Setup::install();
        
        // Varsayılan ayarları oluştur
        if (!get_option('mimoza_default_word_count')) {
            update_option('mimoza_default_word_count', 2000);
        }
        if (!get_option('mimoza_auto_internal_links')) {
            update_option('mimoza_auto_internal_links', 'yes');
        }
        if (!get_option('mimoza_auto_external_links')) {
            update_option('mimoza_auto_external_links', 'yes');
        }
    }

    /**
     * Admin Bildirimleri
     */
    public function show_admin_notices() {
        $screen = get_current_screen();
        
        if (strpos($screen->id, 'mimoza') === false) {
            return;
        }
        
        // API anahtarı kontrolü
        $has_ai_key = get_option('mimoza_openai_api_key') || 
                      get_option('mimoza_claude_api_key') || 
                      get_option('mimoza_gemini_api_key') ||
                      get_option('mimoza_jasper_api_key');
        
        if (!$has_ai_key && $screen->id !== 'mimoza-ai-seo_page_mimoza-settings') {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong>⚠️ Mimoza AI SEO:</strong> Hiçbir AI API anahtarı girilmemiş. 
                    <a href="<?php echo admin_url('admin.php?page=mimoza-settings'); ?>">Ayarlar sayfasından API anahtarlarınızı ekleyin</a>.
                </p>
            </div>
            <?php
        }
    }
}

// Plugin'i başlat
new Mimoza_AI_SEO_Premium();
