<?php
/**
 * Plugin Name: Mimoza İletişim Pro – Drawer + Callback (fix4 ClickOnly + WhatsApp)
 * Description: Telefon, WhatsApp, Messenger + Drawer menü + Beni Ara (WhatsApp/E-posta). Bu sürümde: Hover ile açılma tamamen kaldırıldı, sadece tıklama ile açılır.
 * Version: 1.10.4-dev
 * Author: Onur Kalafat
 * Text Domain: mimoza-iletisim
 * Requires PHP: 7.2
 * Requires at least: 5.8
 * Tested up to: 6.6
 */

if (!defined('ABSPATH')) { exit; }

define('MIMOZA_ILETISIM_PRO_VER', '1.10.4-dev');

/* ---------- Defaults ---------- */
function mimoza_iletisim_pro_defaults() {
    return [
        'phone'            => '',
        'whatsapp'         => '',
        'messenger'        => '',
        'email'            => '',
        'telegram'         => '', // @username
        'address_url'      => '', // maps link
        'brand_color'      => '#007BFF',
        'position'         => 'right', // left|right|center
        'shape'            => 'pill',  // pill|circle|rounded
        'size'             => 'xl',    // sm|md|lg|xl
        'emphasis'         => true,    // pulse/glow
        'show_labels'      => true,
        'enable_prefill'   => true,
        'prefill_template' => 'Merhaba, {site} üzerinden yazıyorum. Sayfa: {title} – {url}',
        'enable_utm'       => true,
        'utm_source'       => 'mimoza-contact',
        'utm_medium'       => 'widget',
        'utm_campaign'     => 'contact-cta',
        'hours_enabled'    => false,
        'hours_days'       => ['mon','tue','wed','thu','fri'],
        'hours_start'      => '09:00',
        'hours_end'        => '19:00',
        'hours_hide_outside'=> false,
        'exclude_contains' => '',
        // Drawer / Callback
        'enable_drawer'    => true,
        'drawer_label'     => 'Bize Ulaşın',
        'enable_callback'  => true,
        'callback_mode'    => 'whatsapp', // whatsapp|email
        'callback_email'   => '',
        'privacy_url'      => '',
        'success_text'     => 'Teşekkürler! En kısa sürede sizi arayacağız.',
    ];
}
function mimoza_iletisim_pro_get_options() {
    $defaults = mimoza_iletisim_pro_defaults();
    $opts = get_option('mimoza_iletisim_pro', []);
    if (!is_array($opts)) { $opts = []; }
    return array_merge($defaults, $opts);
}
function mimoza_iletisims_sanitize_on($val){ return (isset($val) && ($val==='1' || $val==='on')); }

/* ---------- Admin UI ---------- */
function mimoza_iletisim_pro_menu() {
    add_menu_page(
        __('Mimoza İletişim', 'mimoza-iletisim'),
        __('Mimoza İletişim', 'mimoza-iletisim'),
        'manage_options',
        'mimoza-iletisim-pro',
        'mimoza_iletisim_pro_settings_page',
        'dashicons-email-alt',
        100
    );
}
add_action('admin_menu', 'mimoza_iletisim_pro_menu');

function mimoza_iletisim_pro_settings_page() {
    if (!current_user_can('manage_options')) return;
    $opts = mimoza_iletisim_pro_get_options();

    if ($_SERVER['REQUEST_METHOD']==='POST' && check_admin_referer('mimoza_iletisim_pro_save','mimoza_iletisim_pro_nonce')) {
        // Channels
        $opts['phone']       = sanitize_text_field($_POST['phone'] ?? '');
        $opts['whatsapp']    = preg_replace('/\D+/','', $_POST['whatsapp'] ?? '');
        $opts['messenger']   = sanitize_text_field($_POST['messenger'] ?? '');
        $opts['email']       = sanitize_email($_POST['email'] ?? '');
        $opts['telegram']    = sanitize_text_field($_POST['telegram'] ?? '');
        $opts['address_url'] = esc_url_raw($_POST['address_url'] ?? '');
        // Styles
        $opts['brand_color'] = sanitize_hex_color($_POST['brand_color'] ?? '#007BFF');
        $opts['position']    = in_array($_POST['position'] ?? 'right', ['left','right','center'], true) ? $_POST['position'] : 'right';
        $opts['shape']       = in_array($_POST['shape'] ?? 'pill', ['pill','circle','rounded'], true) ? $_POST['shape'] : 'pill';
        $opts['size']        = in_array($_POST['size'] ?? 'xl', ['sm','md','lg','xl'], true) ? $_POST['size'] : 'xl';
        $opts['emphasis']    = mimoza_iletisims_sanitize_on($_POST['emphasis'] ?? '');
        $opts['show_labels'] = mimoza_iletisims_sanitize_on($_POST['show_labels'] ?? '');
        // Prefill & UTM
        $opts['enable_prefill']   = mimoza_iletisims_sanitize_on($_POST['enable_prefill'] ?? '');
        $opts['prefill_template'] = sanitize_text_field($_POST['prefill_template'] ?? '');
        $opts['enable_utm']   = mimoza_iletisims_sanitize_on($_POST['enable_utm'] ?? '');
        $opts['utm_source']   = sanitize_key($_POST['utm_source'] ?? 'mimoza-contact');
        $opts['utm_medium']   = sanitize_key($_POST['utm_medium'] ?? 'widget');
        $opts['utm_campaign'] = sanitize_title($_POST['utm_campaign'] ?? 'contact-cta');
        // Hours
        $opts['hours_enabled']     = mimoza_iletisims_sanitize_on($_POST['hours_enabled'] ?? '');
        $opts['hours_days']        = array_values(array_intersect((array)($_POST['hours_days'] ?? []), ['mon','tue','wed','thu','fri','sat','sun']));
        $opts['hours_start']       = preg_match('/^\d{2}:\d{2}$/', $_POST['hours_start'] ?? '') ? $_POST['hours_start'] : '09:00';
        $opts['hours_end']         = preg_match('/^\d{2}:\d{2}$/', $_POST['hours_end'] ?? '') ? $_POST['hours_end'] : '19:00';
        $opts['hours_hide_outside']= mimoza_iletisims_sanitize_on($_POST['hours_hide_outside'] ?? '');
        // Visibility
        $opts['exclude_contains']  = implode("\n", array_filter(array_map('sanitize_text_field', explode("\n", $_POST['exclude_contains'] ?? ''))));
        // Drawer & Callback
        $opts['enable_drawer']   = mimoza_iletisims_sanitize_on($_POST['enable_drawer'] ?? '');
        $opts['drawer_label']    = sanitize_text_field($_POST['drawer_label'] ?? 'Bize Ulaşın');
        $opts['enable_callback'] = mimoza_iletisims_sanitize_on($_POST['enable_callback'] ?? '');
        $opts['callback_mode']   = in_array($_POST['callback_mode'] ?? 'whatsapp', ['whatsapp','email'], true) ? $_POST['callback_mode'] : 'whatsapp';
        $opts['callback_email']  = sanitize_email($_POST['callback_email'] ?? '');
        $opts['privacy_url']     = esc_url_raw($_POST['privacy_url'] ?? '');
        $opts['success_text']    = sanitize_text_field($_POST['success_text'] ?? '');

        update_option('mimoza_iletisim_pro', $opts);
        echo '<div class="updated"><p>'.esc_html__('Ayarlar kaydedildi.', 'mimoza-iletisim').'</p></div>';
    }

    $admin_email_suggestion = get_option('admin_email');
    ?>
    <div class="wrap">
      <h1><?php esc_html_e('Mimoza İletişim Ayarları','mimoza-iletisim'); ?></h1>
      <form method="post">
        <?php wp_nonce_field('mimoza_iletisim_pro_save','mimoza_iletisim_pro_nonce'); ?>
        <h2 class="title">Kanallar</h2>
        <table class="form-table">
          <tr><th>Telefon</th><td><input type="text" name="phone" value="<?php echo esc_attr($opts['phone']); ?>" class="regular-text" placeholder="+90 ..."></td></tr>
          <tr><th>WhatsApp</th><td><input type="text" name="whatsapp" value="<?php echo esc_attr($opts['whatsapp']); ?>" class="regular-text" placeholder="9053... (sadece rakam)"></td></tr>
          <tr><th>Messenger</th><td><input type="text" name="messenger" value="<?php echo esc_attr($opts['messenger']); ?>" class="regular-text" placeholder="sayfa/kullanici"></td></tr>
          <tr><th>E-posta</th><td><input type="email" name="email" value="<?php echo esc_attr($opts['email']); ?>" class="regular-text" placeholder="iletisim@..."></td></tr>
          <tr><th>Telegram</th><td><input type="text" name="telegram" value="<?php echo esc_attr($opts['telegram']); ?>" class="regular-text" placeholder="@kullanici"></td></tr>
          <tr><th>Adres/Harita</th><td><input type="url" name="address_url" value="<?php echo esc_attr($opts['address_url']); ?>" class="regular-text" placeholder="https://maps.google.com/?q=..."></td></tr>
        </table>

        <h2 class="title">Görsel & Davranış</h2>
        <table class="form-table">
          <tr><th>Marka Rengi</th><td><input type="text" name="brand_color" value="<?php echo esc_attr($opts['brand_color']); ?>" class="regular-text" placeholder="#007BFF"></td></tr>
          <tr><th>Konum</th>
            <td>
              <label><input type="radio" name="position" value="left"  <?php checked($opts['position'],'left');  ?>> Sol</label>&nbsp;
              <label><input type="radio" name="position" value="center"<?php checked($opts['position'],'center');?>> Orta</label>&nbsp;
              <label><input type="radio" name="position" value="right" <?php checked($opts['position'],'right'); ?>> Sağ</label>
            </td>
          </tr>
          <tr><th>Şekil / Boyut / Etiket</th>
            <td>
              <select name="shape">
                <option value="pill" <?php selected($opts['shape'],'pill'); ?>>Pill</option>
                <option value="circle" <?php selected($opts['shape'],'circle'); ?>>Daire</option>
                <option value="rounded" <?php selected($opts['shape'],'rounded'); ?>>Yuvarlatılmış</option>
              </select>
              &nbsp;&nbsp;Boyut:
              <select name="size">
                <option value="sm" <?php selected($opts['size'],'sm'); ?>>Küçük</option>
                <option value="md" <?php selected($opts['size'],'md'); ?>>Orta</option>
                <option value="lg" <?php selected($opts['size'],'lg'); ?>>Büyük</option>
                <option value="xl" <?php selected($opts['size'],'xl'); ?>>XL</option>
              </select>
              &nbsp;&nbsp;<label><input type="checkbox" name="show_labels" <?php checked($opts['show_labels']); ?>> Etiketleri göster</label>
              &nbsp;&nbsp;<label><input type="checkbox" name="emphasis" <?php checked($opts['emphasis']); ?>> Vurguyu etkinleştir</label>
            </td>
          </tr>
          <tr><th>Ön-Dolum (WhatsApp)</th>
            <td>
              <label><input type="checkbox" name="enable_prefill" <?php checked($opts['enable_prefill']); ?>> Aktif</label><br>
              <input type="text" name="prefill_template" value="<?php echo esc_attr($opts['prefill_template']); ?>" class="regular-text" style="width: 600px">
              <p class="description">{title}, {url}, {site}, {date}, {time}</p>
            </td>
          </tr>
          <tr><th>UTM</th>
            <td>
              <label><input type="checkbox" name="enable_utm" <?php checked($opts['enable_utm']); ?>> Aktif</label><br>
              source: <input type="text" name="utm_source" value="<?php echo esc_attr($opts['utm_source']); ?>" size="12">
              medium: <input type="text" name="utm_medium" value="<?php echo esc_attr($opts['utm_medium']); ?>" size="12">
              campaign: <input type="text" name="utm_campaign" value="<?php echo esc_attr($opts['utm_campaign']); ?>" size="16">
            </td>
          </tr>
          <tr><th>Çalışma Saatleri</th>
            <td>
              <label><input type="checkbox" name="hours_enabled" <?php checked($opts['hours_enabled']); ?>> Saat sınırlaması</label>&nbsp;
              <label><input type="checkbox" name="hours_hide_outside" <?php checked($opts['hours_hide_outside']); ?>> Saat dışı gizle</label><br>
              <?php $days = ['mon'=>'Pzt','tue'=>'Sal','wed'=>'Çar','thu'=>'Per','fri'=>'Cum','sat'=>'Cts','sun'=>'Paz'];
              foreach ($days as $k=>$v){
                  printf('<label style="margin-right:8px;"><input type="checkbox" name="hours_days[]" value="%s" %s> %s</label>', esc_attr($k), in_array($k,$opts['hours_days'],true)?'checked':'' , esc_html($v));
              } ?>
              <br> Başlangıç: <input type="text" name="hours_start" value="<?php echo esc_attr($opts['hours_start']); ?>" size="6">
              Bitiş: <input type="text" name="hours_end" value="<?php echo esc_attr($opts['hours_end']); ?>" size="6">
            </td>
          </tr>
          <tr><th>Hariç Tut (URL parçası)</th>
            <td>
              <textarea name="exclude_contains" rows="3" cols="80" placeholder="/iletisim, /gizlilik"><?php echo esc_textarea($opts['exclude_contains']); ?></textarea>
            </td>
          </tr>
        </table>

        <h2 class="title">Drawer Menü & Beni Ara</h2>
        <table class="form-table">
          <tr><th>Drawer (tek uçan buton)</th>
            <td>
              <label><input type="checkbox" name="enable_drawer" <?php checked($opts['enable_drawer']); ?>> Aktif</label><br>
              Buton etiketi: <input type="text" name="drawer_label" value="<?php echo esc_attr($opts['drawer_label']); ?>" class="regular-text" placeholder="Bize Ulaşın">
            </td>
          </tr>
          <tr><th>Beni Ara</th>
            <td>
              <label><input type="checkbox" name="enable_callback" <?php checked($opts['enable_callback']); ?>> Aktif</label>
              <p class="description">Form açıkken, mod seçimine göre WhatsApp yönlendirme ya da e-posta gönderimi yapılır.</p>
              Mod:&nbsp;
              <label><input type="radio" name="callback_mode" value="whatsapp" <?php checked($opts['callback_mode'],'whatsapp'); ?>> WhatsApp</label>&nbsp;
              <label><input type="radio" name="callback_mode" value="email" <?php checked($opts['callback_mode'],'email'); ?>> E-posta</label>
              <br> Bildirim e-postası (yalnızca e-posta modunda): <input type="email" name="callback_email" value="<?php echo esc_attr($opts['callback_email']); ?>" class="regular-text" placeholder="<?php echo esc_attr($admin_email_suggestion); ?>">
              <br> KVKK/Çerez linki: <input type="url" name="privacy_url" value="<?php echo esc_attr($opts['privacy_url']); ?>" class="regular-text" placeholder="https://...">
              <br> Başarı mesajı: <input type="text" name="success_text" value="<?php echo esc_attr($opts['success_text']); ?>" class="regular-text" style="width: 480px">
            </td>
          </tr>
        </table>
        <?php submit_button(); ?>
      </form>
    </div>
    <?php
}

/* ---------- Visibility helper ---------- */
function mimoza_iletisim_pro_visibility_allowed($opts){
    $ex = array_filter(array_map('trim', explode("\n", (string)$opts['exclude_contains'])));
    if (empty($ex)) return true;
    $uri = $_SERVER['REQUEST_URI'] ?? '';
    foreach ($ex as $needle){
        if ($needle !== '' && strpos($uri, $needle) !== false) return false;
    }
    return true;
}

/* ---------- Assets ---------- */
function mimoza_iletisim_pro_enqueue_assets() {
    $url = plugin_dir_url(__FILE__);
    wp_enqueue_style('mimoza-iletisim-pro', $url . 'mimoza-iletisim.css', [], MIMOZA_ILETISIM_PRO_VER);
    wp_enqueue_script('mimoza-iletisim-pro', $url . 'mimoza-iletisim.js', [], MIMOZA_ILETISIM_PRO_VER, true);

    $opts = mimoza_iletisim_pro_get_options();
    $client = [
        'prefillTemplate' => $opts['prefill_template'],
        'enablePrefill'   => (bool) $opts['enable_prefill'],
        'enableUTM'       => (bool) $opts['enable_utm'],
        'utm'             => [
            'source'   => $opts['utm_source'],
            'medium'   => $opts['utm_medium'],
            'campaign' => $opts['utm_campaign'],
        ],
        'hours' => [
            'enabled' => (bool) $opts['hours_enabled'],
            'days'    => $opts['hours_days'],
            'start'   => $opts['hours_start'],
            'end'     => $opts['hours_end'],
            'hideOutside' => (bool) $opts['hours_hide_outside'],
        ],
        'labels'        => (bool) $opts['show_labels'],
        'position'      => $opts['position'],
        'shape'         => $opts['shape'],
        'size'          => $opts['size'],
        'emphasis'      => (bool) $opts['emphasis'],
        'drawer'        => (bool) $opts['enable_drawer'],
        'drawerLabel'   => $opts['drawer_label'],
        'brandColor'    => $opts['brand_color'],
        'callback'      => [
            'enabled'    => (bool) $opts['enable_callback'],
            'mode'       => $opts['callback_mode'],
            'privacyUrl' => $opts['privacy_url'],
            'success'    => $opts['success_text'],
            'waRecipient'=> preg_replace('/\D+/','', $opts['whatsapp']),
        ],
        'ajax'          => [
            'url'   => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('mimoza_cb'),
            'rest'  => esc_url_raw( rest_url('mimoza/v1/nonce') ),
        ]
    ];
    wp_localize_script('mimoza-iletisim-pro', 'MimozaIletisim', $client);
}
add_action('wp_enqueue_scripts', 'mimoza_iletisim_pro_enqueue_assets');

/* ---------- Render ---------- */
function mimoza_iletisim_pro_render_buttons_list($opts){
    $labels = $opts['show_labels'];
    $out = '';
    $make = function($class, $href, $label, $icon) use ($labels){
        $lab = $labels ? '<span class="label">'. esc_html($label) .'</span>' : '';
        return sprintf('<a class="contact-button %1$s" href="%2$s" rel="nofollow noopener">%4$s%3$s</a>',
            esc_attr($class), esc_url($href), $lab, '<span class="icon" aria-hidden="true">'.$icon.'</span>');
    };
    if ($opts['phone']){
        $out .= $make('contact-phone', 'tel:'. preg_replace('/\s+/', '', $opts['phone']), 'Ara', '📞');
    }
    if ($opts['whatsapp']){
        $out .= sprintf('<a class="contact-button contact-whatsapp" data-wa="%1$s" href="%2$s" rel="nofollow noopener"><span class="icon">💬</span>%3$s</a>',
            esc_attr(preg_replace('/\D+/','', $opts['whatsapp'])), esc_url('https://wa.me/'. preg_replace('/\D+/','', $opts['whatsapp'])), $labels?'<span class="label">WhatsApp</span>':'');
    }
    if ($opts['messenger']){
        $out .= sprintf('<a class="contact-button contact-messenger" data-msgr="%1$s" href="%2$s" rel="nofollow noopener"><span class="icon">📩</span>%3$s</a>',
            esc_attr($opts['messenger']), esc_url('https://m.me/'. ltrim($opts['messenger'],'/')), $labels?'<span class="label">Messenger</span>':'');
    }
    if ($opts['email']){
        $out .= $make('contact-email', 'mailto:'. esc_attr($opts['email']), 'E-posta', '✉️');
    }
    if ($opts['telegram']){
        $tg = ltrim($opts['telegram'], '@');
        $out .= $make('contact-telegram', 'https://t.me/'. $tg, 'Telegram', '📨');
    }
    if ($opts['address_url']){
        $out .= $make('contact-address', $opts['address_url'], 'Yol Tarifi', '📍');
    }
    return $out;
}

function mimoza_iletisim_pro_render() {
    $opts = mimoza_iletisim_pro_get_options();
    if (!mimoza_iletisim_pro_visibility_allowed($opts)) return;

    $classes = [
        'mimoza-contact',
        'is-' . esc_attr($opts['position']),
        'shape-' . esc_attr($opts['shape']),
        'size-' . esc_attr($opts['size']),
        $opts['emphasis'] ? 'is-emphasis' : '',
        $opts['show_labels'] ? 'has-labels' : 'no-labels'
    ];
    $style = sprintf('--mimoza-brand:%s;', esc_attr($opts['brand_color']));

    echo '<div class="'. esc_attr(implode(' ', array_filter($classes))) .'" style="'. esc_attr($style) .'">';

    if (!empty($opts['enable_drawer'])){
        // FAB + Drawer (click only)
        echo '<button class="mimoza-fab" id="mimoza-fab" aria-expanded="false" aria-controls="mimoza-drawer" aria-label="'. esc_attr($opts['drawer_label']) .'">';
        echo '<span class="fab-icon">✆</span><span class="fab-label">'. esc_html($opts['drawer_label']) .'</span>';
        echo '</button>';

        echo '<div class="mimoza-backdrop" id="mimoza-backdrop" hidden></div>';
        echo '<div class="mimoza-drawer" id="mimoza-drawer" role="dialog" aria-modal="true" aria-labelledby="mimoza-drawer-title" hidden>';
        echo '<div class="drawer-header"><div class="drawer-title" id="mimoza-drawer-title">'. esc_html($opts['drawer_label']) .'</div>';
        echo '<button class="drawer-close" id="mimoza-drawer-close" aria-label="Kapat">×</button></div>';

        echo '<div class="drawer-body">';
        echo '<div class="drawer-channels">'. mimoza_iletisim_pro_render_buttons_list($opts) .'</div>';

        if (!empty($opts['enable_callback'])){
            $privacy = '';
            if (!empty($opts['privacy_url'])){
                $privacy = ' – <a href="'. esc_url($opts['privacy_url']) .'" target="_blank" rel="noopener">KVKK</a>';
            }
            echo '<div class="drawer-sep"></div>';
            echo '<form class="mimoza-callback" id="mimoza-callback" novalidate>
                    <div class="cb-row">
                      <input type="text" name="name" placeholder="Ad Soyad" autocomplete="name" required>
                    </div>
                    <div class="cb-row">
                      <input type="tel" name="phone" placeholder="Telefon" autocomplete="tel" required>
                    </div>
                    <div class="cb-row">
                      <textarea name="note" rows="2" placeholder="Not (opsiyonel)"></textarea>
                    </div>
                    <div class="cb-row small">
                      <label class="cb-privacy">
                        <input type="checkbox" name="agree" required>
                        <span>Bilgilerimin sizi aramak için kullanılmasını onaylıyorum'. $privacy .'</span>
                      </label>
                    </div>
                    <input type="hidden" name="hpt" value="">
                    <input type="hidden" name="ts" value="'. time() .'">
                    <button type="submit" class="cb-submit">Beni Ara</button>
                    <div class="cb-msg" aria-live="polite"></div>
                  </form>';
        }

        echo '</div>'; // body
        echo '</div>'; // drawer
    } else {
        // Classic stacked buttons (legacy)
        echo '<div class="mimoza-contact-buttons">'. mimoza_iletisim_pro_render_buttons_list($opts) .'</div>';
    }
    echo '</div>'; // wrap
}
add_action('wp_footer', 'mimoza_iletisim_pro_render');
add_shortcode('mimoza_iletisim', 'mimoza_iletisim_pro_render');

/* ---------- AJAX: Callback (email mode) ---------- */
function mimoza_iletisim_pro_rate_key(){
    $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    return 'mimoza_cb_'. md5($ip);
}
function mimoza_iletisim_pro_callback_handler(){
    check_ajax_referer('mimoza_cb','nonce');
    $opts = mimoza_iletisim_pro_get_options();
    if (($opts['callback_mode'] ?? 'whatsapp') !== 'email'){
        wp_send_json_error(['msg'=>'E-posta modu kapalı.'], 400);
    }
    // Simple rate limit (60s per IP)
    $key = mimoza_iletisim_pro_rate_key();
    if (get_transient($key)) {
        wp_send_json_error(['msg'=>'Lütfen biraz sonra tekrar deneyin.'], 429);
    }

    $name  = sanitize_text_field($_POST['name'] ?? '');
    $phone = preg_replace('/[^\d\+\s]/','', $_POST['phone'] ?? '');
    $note  = sanitize_textarea_field($_POST['note'] ?? '');
    $agree = !empty($_POST['agree']);
    $hpt   = trim($_POST['hpt'] ?? '');
    $ts    = intval($_POST['ts'] ?? 0);

    if ($hpt !== '') { wp_send_json_error(['msg'=>'Spam engellendi.'], 400); }
    if (time() - $ts < 3) { wp_send_json_error(['msg'=>'Çok hızlı gönderim.'], 400); }
    if (!$agree) { wp_send_json_error(['msg'=>'Lütfen onay kutusunu işaretleyin.'], 400); }
    if (strlen($name) < 2 || strlen($phone) < 6) { wp_send_json_error(['msg'=>'Ad ve telefon zorunludur.'], 400); }

    // Compose email
    $to = $opts['callback_email'] ? $opts['callback_email'] : get_option('admin_email');
    $subject = sprintf('Geri Arama Talebi – %s', wp_parse_url(home_url(), PHP_URL_HOST));
    $page  = esc_url_raw($_POST['page'] ?? '');
    $title = sanitize_text_field($_POST['title'] ?? '');
    $ua    = sanitize_text_field($_SERVER['HTTP_USER_AGENT'] ?? '');

    $body = "Ad Soyad: {$name}
Telefon: {$phone}
Not: {$note}

Sayfa: {$title}
URL: {$page}
Tarih: ". date('Y-m-d H:i:s') ."
UA: {$ua}
";
    $headers = ['Content-Type: text/plain; charset=UTF-8'];

    $ok = wp_mail($to, $subject, $body, $headers);
    if ($ok){
        set_transient($key, 1, 60);
        do_action('mimoza_iletisim_pro_callback_success', compact('name','phone','note','page'));
        wp_send_json_success(['msg'=> $opts['success_text'] ?: 'Talebiniz alındı.']);
    }else{
        wp_send_json_error(['msg'=>'Gönderim başarısız. Lütfen farklı bir kanalı deneyin.'], 500);
    }
}
add_action('wp_ajax_mimoza_cb', 'mimoza_iletisim_pro_callback_handler');
add_action('wp_ajax_nopriv_mimoza_cb', 'mimoza_iletisim_pro_callback_handler');

/* ---------- REST: Nonce refresh ---------- */
function mimoza_iletisim_pro_register_rest(){
    register_rest_route('mimoza/v1', '/nonce', [
        'methods'  => 'GET',
        'callback' => function(){ return ['nonce' => wp_create_nonce('mimoza_cb')]; },
        'permission_callback' => '__return_true',
    ]);
}
add_action('rest_api_init','mimoza_iletisim_pro_register_rest');
